function _typeof(o) { "@babel/helpers - typeof"; return _typeof = "function" == typeof Symbol && "symbol" == typeof Symbol.iterator ? function (o) { return typeof o; } : function (o) { return o && "function" == typeof Symbol && o.constructor === Symbol && o !== Symbol.prototype ? "symbol" : typeof o; }, _typeof(o); }
/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * Defines {@link baja.comm.Callback}.
 * @module baja/comm/Callback
 * @private
 */
define(["bajaScript/sys", "bajaScript/baja/sys/BaseBajaObj", "bajaScript/baja/comm/Batch"], function (baja, BaseBajaObj, Batch) {
  "use strict";

  var subclass = baja.subclass,
    callSuper = baja.callSuper,
    Callback;

  /**
   * A Callback object has ok and fail methods. It is used to make a comms call in
   * BajaScript. When a comms call is made, additional callbacks may be needed to
   * process the incoming data before calling the original ok or fail callbacks are 
   * executed. Therefore, extra help functions have been added to this object
   * to for convenience.
   * 
   * This method should only be used internally by BajaScript. It should never appear
   * in the public API.
   *
   * @class
   * @alias baja.comm.Callback
   * @extends baja.BaseBajaObj
   * @private
   */
  Callback = function Callback(ok, fail, batch) {
    var that = this,
      doOk = ok || baja.ok,
      doFail = fail || baja.fail;
    callSuper(Callback, that, arguments);
    that.$batch = batch || new Batch();
    that.$batchNext = baja.noop;
    that.ok = function () {
      // Invoke user's ok callback and then make a batch callback
      try {
        doOk.apply(this, arguments);
        that.onOk(this, arguments);
      } finally {
        that.$batchNext();
      }
    };
    that.fail = function () {
      // Invoke the user's fail callback and then make a batch callback
      try {
        doFail.apply(this, arguments);
        that.onFail(this, arguments);
      } finally {
        that.$batchNext();
      }
    };

    // Make a note of whether the batch was originally defined
    that.$orgBatchDef = _typeof(batch) === "object";
  };
  subclass(Callback, BaseBajaObj);

  /**
   * Add an 'ok' callback.
   * 
   * When calling some BajaScript that causes a network call, there will be 
   * other callbacks that need to be processed before the original 'user' 
   * callbacks are invoked. Therefore, 'addOk' and 'addFail' can be used to 
   * chain up extra Callbacks if needed.
   * 
   * Please note, that when adding an intermediate calllback, the 
   * `ok` or `fail` method passed in **must** be called at some point.
   * 
   * This method is also extremely useful for intermediate callbacks that 
   * need to make other asynchronous network calls before calling the user's 
   * original 'ok' or 'fail' callback functions.
   *
   * @private
   *
   * @see baja.comm.Callback#fail
   *
   * @param {Function} newOk  the callback Function. This function must accept three
   *                          arguments including the current ok and fail function as well
   *                          as any arguments specific to the callback function.
   *                          By convention, the 'ok' or 'fail' functions passed in must be called
   *                          after the new callback has finished.
   *                          
   * @example
   *   // Create a callback object with user's original ok and fail function callbacks...
   *   var cb = new baja.comm.Callback(ok, fail, batch);
   *
   *   // Add an intermediate callback...
   *   cb.addOk(function (ok, fail, resp) {
   *     // Process the response 'resp' object...
   *     var val = processResponse(resp);
   *     
   *     // Now call 'ok' callback passed in with processed response...
   *     ok(val);
   *   });
   */
  Callback.prototype.addOk = function (newOk) {
    // Make a reference to the current ok and fail callback functions
    var currentOk = this.ok,
      currentFail = this.fail;

    // Create a new 'ok' callback closure. When invoked, the new 'ok' function will be called
    // with the previous 'ok' and 'fail' functions passed in as arguments to the beginning of
    // the function. By convention, the new callback ok function should invoke the 'ok' function
    // passed into it as an argument (or fail if there is a problem with the data).
    this.ok = function () {
      var args = Array.prototype.slice.call(arguments);
      args.splice(0, 0, currentOk, currentFail);
      newOk.apply(this, args);
    };
  };

  /**
   * Add a 'fail' callback.
   * 
   * When calling some BajaScript that causes a network call, there will be other callbacks
   * that need to be processed before the original 'user' callbacks are invoked. Therefore,
   * `addOk` and `addFail` can be used to chain up extra Callbacks if needed.
   * 
   * Please note, that when adding an intermediate callback, the `ok` or 
   * `fail` method passed in **must** be called at some point.
   *
   * @private
   *
   * @see baja.comm.Callback#fail
   *
   * @param {Function} newFail  the callback Function. This function must accept three
   *                            arguments including the current ok and fail function as well
   *                            as any arguments specific to the callback function.
   *                            By convention, the 'ok' or 'fail' functions passed in must be called
   *                            after the new callback has finished.
   * 
   * @example
   *   // Create a callback object with user's original ok and fail function callbacks...
   *   var cb = new baja.comm.Callback(ok, fail, batch);
   *
   *   // Add an intermediate callback...
   *   cb.addFail(function (ok, fail, err) {
   *     // Process the error messages before calling the original 'fail' callback...
   *     var niceMsg = processError(err);
   *     
   *     // Now call 'fail' callback passed in with processed error message...
   *     fail(niceMsg);
   *   });
    */
  Callback.prototype.addFail = function (newFail) {
    var currentOk = this.ok,
      currentFail = this.fail;
    this.fail = function () {
      var args = Array.prototype.slice.call(arguments);
      args.splice(0, 0, currentOk, currentFail);
      newFail.apply(this, args);
    };
  };

  /**
   * Return true if the batch object was originally defined when the batch was created.
   * 
   * @private
   *
   * @returns {Boolean}
   */
  Callback.prototype.isOrgBatchDef = function () {
    return this.$orgBatchDef;
  };

  /**
   * Add a request to the callback's batch object.
   * 
   * @private
   *
   * @see baja.comm.Batch#addReq
   *
   * @param {String} channel
   * @param {String} key
   * @param body
   * @param {Boolean} [queueOnCommit]
   */
  Callback.prototype.addReq = function (channel, key, body, queueOnCommit) {
    this.$batch.addReq(channel, key, body, this, queueOnCommit);
  };

  /**
   * If a Batch object was not originally passed in when this object was
   * created then commit. Otherwise do nothing.
   * 
   * Therefore, if a Batch was originally passed into this Callback
   * object, it is assumed the caller will invoke the Batch commit method
   * when appropriate (i.e. they've batched up all of the network requests
   * that are going to be made).
   *
   * @private
   *
   * @see baja.comm.Batch#commit
   * @see baja.comm.Callback#isOrgBatchDef
   */
  Callback.prototype.autoCommit = function () {
    // If there was a batch object originally defined then don't commit
    // as the batch will be committed elsewhere
    if (!this.isOrgBatchDef()) {
      this.commit();
    }
  };

  /**
   * Commit the this object's Batch.
   * 
   * This asynchronously makes a network call.
   * 
   * @private
   *
   * @see baja.comm.Batch#commit
   */
  Callback.prototype.commit = function () {
    this.$batch.commit();
  };

  /** 
   * Return this object's batch object.
   * 
   * @private
   *
   * @returns {baja.comm.Batch}
   */
  Callback.prototype.getBatch = function () {
    return this.$batch;
  };

  /**
   * Invoked once the callback has successfully completed.
   *
   * This method is overridden per environment (i.e. browser, node).
   *
   * @private
   * @function
   *
   * @param cx
   * @param args
   */
  Callback.prototype.onOk = baja.noop;

  /**
   * Invoked once the callback has failed.
   *
   * This method is overridden per environment (i.e. browser, node).
   *
   * @private
   * @function
   *
   * @param cx
   * @param args
   */
  Callback.prototype.onFail = baja.noop;

  /**
   * Return a promise to the callback.
   *
   * This method is overridden per environment (i.e. browser, node).
   *
   * @private
   *
   * @returns promise
   */
  Callback.prototype.promise = function () {
    throw new Error("Callback promises not implemented");
  };
  return Callback;
});
