/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * @module baja/boxcs/AddOp
 * @private
 */
define(["bajaScript/bson", "bajaScript/baja/comp/Flags", "bajaScript/baja/obj/Facets", "bajaScript/baja/boxcs/SyncOp"], function (baja, Flags, Facets, SyncOp) {
  "use strict";

  var subclass = baja.subclass,
    callSuper = baja.callSuper,
    bajaDef = baja.def;
  var serverDecodeContext = baja.$serverDecodeContext;

  /**
   * @class Add SyncOp.
   *
   * @name AddOp
   * @extends SyncOp
   * @inner
   * @private
   *
   * @param comp the Component the add is happening upon.
   * @param {String} slotName the name of the slot being added.
   * @param val the value for the add operation.
   * @param {Number} flags the slot facets.
   * @param {baja.Facets} facets the slot facets.
   */
  function AddOp(comp, slotName, val, flags, facets) {
    callSuper(AddOp, this, arguments);

    // TODO: What about getting the name of the Property that was created 
    // from the Server???

    // Encode argument to a data structure
    var a = {
      nm: this.getId(),
      h: comp.getHandle(),
      b: baja.bson.encodeValue(val) // Encode the value to BSON
    };
    if (slotName !== null) {
      a.n = slotName;
    }
    if (flags !== 0) {
      a.f = Flags.encodeToString(flags);
    }
    if (facets !== null && facets !== Facets.DEFAULT) {
      a.facets = facets.encodeToString();
    }
    this.$arg = a;
  }
  subclass(AddOp, SyncOp);
  AddOp.id = "a";

  /**
   * Decode and commit the SyncOp.
   *
   * @private
   *
   * @param {baja.Component} comp the Component being added to.
   * @param {object} sp the syncOp data structure to be decoded.
   * @returns {Promise|*} if add is performed, return promise to be resolved when it is complete
   */
  AddOp.decodeAndCommit = function (comp, sp) {
    // TODO: Shouldn't need to add the 'get' check on the end of this if statement but it'll
    // have to do for now. Really the Server should keep track of what Components the client
    // has loaded instead of just firing everything down it gets!

    if (comp && !comp.get(sp.n)) {
      var slot = sp.n,
        displayName = sp.dn,
        bson = sp.b,
        flags = sp.f,
        facets = sp.facets;
      var display = bson.d;
      return baja.bson.decodeAsync(bson, serverDecodeContext).then(function (value) {
        // Perform Component add with Commit Context
        return comp.add({
          slot: slot,
          value: value,
          flags: Flags.decodeFromString(bajaDef(flags, "")),
          facets: Facets.DEFAULT.decodeFromString(bajaDef(facets, ""), baja.Simple.$unsafeDecode),
          cx: {
            commit: true,
            displayName: displayName,
            display: display,
            serverDecode: true
          }
        });
      });
    }
  };
  return AddOp;
});
