/**
 * @file Gradient utilities.
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

define([], function gradientUtil() {
  "use strict";

  /**
   * API Status: **Private**
   * @exports nmodule/gx/rc/util/gradientUtils
   */
  var exports = {};

  /**
   * @param {Number} angle
   * @param {Array} stops
   * @return {String} something like `(45deg, red 0%, orange 25%, yellow 100%)`
   * - note no prefix
   */
  function makeLinearGradient(angle, stops) {
    var css,
      stop,
      len = stops.length,
      i = 0;
    css = '(' + angle + 'deg, ';
    while (i < len) {
      stop = stops[i];
      css += stop[1].toCssString() + ' ' + stop[0];
      if (++i < len) {
        css += ', ';
      }
    }
    css += ')';
    return css;
  }

  /**
   * @param {string} radius
   * @param {String} center
   * @param {Array} stops
   * @return {String} something like `(0.5em circle, red 0%, orange 25%, yellow 100%)`
   * - note no prefix
   */
  function makeRadialGradient(radius, center, stops) {
    var css,
      stop,
      len = stops.length,
      i = 0;
    css = '(' + radius + ' at ' + center + ', ';
    while (i < len) {
      stop = stops[i];
      css += stop[1].toCssString() + ' ' + stop[0];
      if (++i < len) {
        css += ', ';
      }
    }
    css += ')';
    return css;
  }
  function parseStops(stops) {
    var parsed = [],
      stop,
      i;
    for (i = 0; i < stops.length; i++) {
      stop = stops[i];
      parsed[i] = [stop[0], stop[1]];
    }
    return parsed;
  }

  /**
   * @param {Number} angle in degrees. This is a NIAGARA angle like you would
   * find on BBrush.LinearGradient, NOT a CSS angle. A Niagara angle works
   * like a Cartesian plane, where 0 degrees goes straight right and 90 degrees
   * goes straight up. A CSS angle works like a clock, where 0 degrees goes
   * straight up and 90 degrees goes straight right. This Niagara angle will
   * be converted to a CSS angle.
   * @param {Array} stops a two dimensional array of stop percentages and
   * colors. e.g.: `[["0%", "red"], ["25%", "orange"], ["100%", "yellow"]]`
   *
   * @return {String} a `linearGradient` gradient CSS that follows the 
   * unprefixed spec (anglesstart vertical and go clockwise). 
   * DOES NOT include `linear-gradient` prefix
   *
   * or null, if no stops are given.
   */
  exports.createLinearGradientCss = function (angle, stops) {
    if (!stops || !stops.length) {
      return null;
    }
    stops = parseStops(stops);
    return makeLinearGradient(90 - angle, stops);
  };

  /**
   * @param {String} radius
   * @param {String} center of the form eg. "45% 45%", the x and y as a 
   * percentage of the bounding box size.
   * @param {Array} stops a two dimensional array of stop percentages and
   * colors. e.g.: `[["0%", "red"], ["25%", "orange"], ["100%", "yellow"]]`
   *
   * @return {String} a `radialGradient` gradient CSS. 
   * DOES NOT include `radial-gradient` prefix
   *
   * or null, if no stops are given.
   */
  exports.createRadialGradientCss = function (radius, center, stops) {
    if (!stops || !stops.length) {
      return null;
    }
    stops = parseStops(stops);
    return makeRadialGradient(radius, center, stops);
  };
  return exports;
});
