/**
 * @copyright 2019 Tridium, Inc. All Rights Reserved.
 * @author Andy Sutton
 */

define(['baja!', 'Promise', 'underscore', 'nmodule/webEditors/rc/wb/mgr/model/columns/IconMgrColumn', 'nmodule/webEditors/rc/wb/mgr/model/columns/NameMgrColumn', 'nmodule/tagdictionary/rc/column/TagIdColumn', 'nmodule/tagdictionary/rc/column/TagTypeColumn', 'nmodule/webEditors/rc/wb/mgr/model/MgrColumn', 'nmodule/webEditors/rc/wb/table/model/Column', 'nmodule/webEditors/rc/wb/table/tree/TreeTableModel', 'nmodule/webEditors/rc/wb/tree/TreeNode', 'nmodule/tagdictionary/rc/util/taggingUtil', 'nmodule/tagdictionary/rc/taginfo/BaseTagInfoMixin', 'nmodule/tagdictionary/rc/taginfo/TagGroupInfoMixin', 'lex!tagdictionary', 'log!nmodule.tagdictionary.rc.model.AvailableTagModel'], function (baja, Promise, _, IconMgrColumn, NameMgrColumn, TagIdColumn, TagTypeColumn, MgrColumn, Column, TreeTableModel, TreeNode, taggingUtil, BaseTagInfoMixin, TagGroupInfoMixin, lexs, log) {
  'use strict';

  var tagdictionaryLex = lexs[0],
    logError = log.severe.bind(log);
  var TAGS_FOLDER_NAME = 'TagsFolder',
    TAG_GROUPS_FOLDER_NAME = 'TagGroupsFolder',
    TAG_FILTER_VALID_ONLY = taggingUtil.TAG_FILTER_VALID_ONLY,
    TAG_FILTER_BEST_ONLY = taggingUtil.TAG_FILTER_BEST_ONLY,
    TAG_FOLDER_ICON = taggingUtil.TAG_FOLDER_ICON;

  // A standard NameMgrColumn with the displayName overridden
  var TAG_NAME_COL = new NameMgrColumn();
  TAG_NAME_COL.toDisplayName = function () {
    return tagdictionaryLex.getSafe('tagManager.tag');
  };
  var COLUMNS = [new IconMgrColumn(), TAG_NAME_COL, new TagIdColumn({
    displayName: tagdictionaryLex.getSafe('tagManager.tagId'),
    flags: Column.flags.UNSEEN
  }), new TagTypeColumn({
    displayName: tagdictionaryLex.getSafe('tagManager.tagType')
  })];

  /**
   * API Status: **Private**
   * @exports nmodule/tagdictionary/rc/model/AvailableTagModel
   */
  var exports = {};

  /**
   * @returns {Promise.<module:nmodule/webEditors/rc/wb/table/tree/TreeTableModel>}
   */
  exports.make = function () {
    return TreeTableModel.make({
      columns: COLUMNS
    });
  };

  /**
   * Populate the manager's learn table with available tags.
   *
   * @param {module:nmodule/tagdictionary/rc/TagUxManager} manager
   * @returns {Promise}
   */
  exports.populateLearnTable = function (manager) {
    var model = manager.getLearnModel(),
      tagInfoDiscoveries = manager.$tagInfos,
      tagGroupInfoDiscoveries = manager.$tagGroupInfos,
      filterParams = manager.$discoveryTableFilterParams,
      showValidOnly = filterParams.showAllValidBest === TAG_FILTER_VALID_ONLY.getOrdinal(),
      showIdealOnly = filterParams.showAllValidBest === TAG_FILTER_BEST_ONLY.getOrdinal(),
      textFilter = filterParams.textFilter;

    // set defaults for expanding the tags/tag groups folders using the restored values if present.
    var expandTags = manager.$discoveryTableFolders && manager.$discoveryTableFolders.tagsExpanded || false,
      expandTagGroups = manager.$discoveryTableFolders && manager.$discoveryTableFolders.tagGroupsExpanded || false;
    function include(baseTagInfoMixin) {
      if (showValidOnly && !baseTagInfoMixin.isValidFor() || showIdealOnly && !baseTagInfoMixin.isIdealFor()) {
        return false;
      }
      if (textFilter) {
        return baseTagInfoMixin.getTagId().getName().match(new RegExp(textFilter, 'gi'));
      }
      return true;
    }

    // Tags
    var tagNodes = _.chain(tagInfoDiscoveries).filter(include).map(function (tagInfo) {
      return taggingUtil.makeTagTableNode(tagInfo, false, baja.Facets.DEFAULT);
    }).value();

    // Tag Groups
    var tagGroupNodes = _.chain(tagGroupInfoDiscoveries).filter(include).map(function (tagGroupInfo) {
      return taggingUtil.makeTagGroupLearnTableNode(tagGroupInfo);
    }).value();
    var dictionaryHasTagGroups = tagGroupInfoDiscoveries.length > 0;

    // update the values for expanding the tags/tag groups folders based on the current model.
    model.getRows().forEach(function (row) {
      if (row.getTreeNode().getName() === TAGS_FOLDER_NAME) {
        expandTags = model.isExpanded(row);
      } else if (row.getTreeNode().getName() === TAG_GROUPS_FOLDER_NAME) {
        expandTagGroups = model.isExpanded(row);
      }
    });
    var tagsFolderNode = makeFolder(TAGS_FOLDER_NAME, tagdictionaryLex.getSafe('tag.objects'), tagNodes);
    var tagGroupsFolderNode = dictionaryHasTagGroups ? makeFolder(TAG_GROUPS_FOLDER_NAME, tagdictionaryLex.getSafe('tagGroup.objects'), tagGroupNodes) : null;
    var rowsToInsert = [tagsFolderNode];
    if (tagGroupsFolderNode) {
      rowsToInsert.push(tagGroupsFolderNode);
    }
    return model.clearRows().then(function () {
      return model.insertRows(rowsToInsert, 0);
    }).then(function () {
      var expandPromises = [];
      model.getRows().forEach(function (row) {
        if (expandTags && row.getTreeNode().getName() === TAGS_FOLDER_NAME) {
          expandPromises.push(model.expand(row));
        }
        if (expandTagGroups && row.getTreeNode().getName() === TAG_GROUPS_FOLDER_NAME) {
          expandPromises.push(model.expand(row));
        }
      });
      return Promise.all(expandPromises);
    });
  };

  /**
   * Save whether the folders are expanded.
   *
   * @param {module:nmodule/tagdictionary/rc/TagUxManager} manager
   * @returns {Object} an object with the state to be persisted
   */
  exports.saveState = function (manager) {
    var model = manager.getLearnModel();
    var tagsExpanded = false,
      tagGroupsExpanded = false;
    model.getRows().forEach(function (row) {
      if (row.getTreeNode().getName() === TAGS_FOLDER_NAME) {
        tagsExpanded = model.isExpanded(row);
      } else if (row.getTreeNode().getName() === TAG_GROUPS_FOLDER_NAME) {
        tagGroupsExpanded = model.isExpanded(row);
      }
    });
    return {
      tagsExpanded: tagsExpanded,
      tagGroupsExpanded: tagGroupsExpanded
    };
  };

  /**
   * Update manager row styling - add hover text for Tag Groups
   *
   * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row
   * @param {JQuery} dom   * @param row@param {JQuery} dom
   * @param {baja.Subscriber} sub   * @param row@param {JQuery} dom
   * @returns {JQuery} dom element with updated styling
   */
  exports.finishTableRow = function (row, dom, sub) {
    var treeNode = row.getTreeNode();
    if (treeNode.isTagGroup && treeNode.isTagGroup()) {
      dom.on('mouseover', function () {
        var subject = row.getSubject();
        taggingUtil.getTagsInTagGroup(treeNode.getName(), subject.getDefaultValue(), sub).then(function (tagGroupsStr) {
          dom.attr('title', tagGroupsStr);
        })["catch"](logError);
        return false;
      });
    }
    return dom;
  };

  /*
   * Make a container component for TagInfo or TagGroupInfo objects.
   *
   * @param  {String} name
   * @param  {String} displayName
   * @param  {Array.<module:nmodule/tagdictionary/rc/taginfo/BaseTagInfoMixin>} kids
   * @returns {module:nmodule/webEditors/rc/wb/tree/TreeNode}
   */
  var makeFolder = function makeFolder(name, displayName, kids) {
    var folder = new TreeNode(name, displayName, kids);
    var tagFolderComp = baja.$('baja:Component');
    // override #getName & #getDisplayName
    tagFolderComp.getName = _.constant(name);
    tagFolderComp.getDisplayName = _.constant(displayName);
    folder.value = _.constant(tagFolderComp);
    folder.getIcon = _.constant(TAG_FOLDER_ICON);
    folder.mayHaveKids = _.constant(true);
    folder.isGroup = _.constant(true);
    return folder;
  };
  return exports;
});
