/*
Using Field Editors
===================

The <any> tag instructs spandrel to build whatever editor is appropriate for the
given value type. This is the behavior in which we request an editor for a
String and receive a StringEditor; a BooleanEditor for a boolean, and so on.

However, spandrel itself does not know what Niagara is. In Niagara world, field
editors are registered as agents on Niagara types via the registry. For spandrel
to construct the right field editors for Niagara values, we must give it the
information it needs to look up field editors "the Niagara way."

Fortunately, all it needs is a hint. In this example, note the second object
given to the spandrel function. This can include additional metadata to
customize our spandrel widget's behavior. The "strategy: 'niagara'" option is
all spandrel needs: passing that will cause your spandrel widget to use Niagara
to perform widget lookups. (If you've used fe.js from the webEditors module,
it's the same behavior.) There are other methods of customizing spandrel's
lookup behavior, but "strategy: 'niagara'" is the 99% use case.

As in fe.js, there are two ways to tell spandrel to dynamically construct a
field editor.

Giving the <any> tag a "value" attribute will cause it to construct a standalone
editor for that value. You are responsible for reading and saving any
user-entered changes.

Giving it "complex" and "slot" will cause it to construct a field editor that
knows how to save changes back to the parent Complex.

The example below will construct a minimal "Property Table" that demonstrates
both of the usages. (This widget can also rename the component, which the usual
Property Sheet cannot.)
*/

/** @jsx spandrel.jsx */

define([
  'baja!',
  'bajaux/spandrel',
  'bajaux/Widget',
  'Promise' ], function (
  baja,
  spandrel,
  Widget,
  Promise) {

  'use strict';

  class PropertyTable extends spandrel((comp, { self }) => {
    return [
      <div>
        <label>Component name:</label>
        {/* just passing "value" means that we are responsible for reading out
          and saving user-entered changes. */}
        <any tagName="span"
             className="componentName"
             value={ baja.SlotPath.unescape(comp.getName()) }
             formFactor="mini" />
      </div>,
      <table className="ux-table">
        <thead className="ux-table-head">
        <tr className="ux-table-row">
          <th>Name</th><th>Value</th>
        </tr>
        </thead>
        <tbody>
        {
          comp.getSlots().properties().toArray().map((prop) => (
            <tr className="ux-table-row">
              <td>{ prop.getName() }</td>
              <td>
                {/* passing "complex" and "slot" gives us a field editor that
                 write changes to the complex when it itself is saved. */}
                <any complex={ comp } slot={ prop } formFactor="mini" className="fe" />
              </td>
            </tr>
          ))
        }
        </tbody>
      </table>,
      <button type="button" onClick={() => self.save()}>
        Save
      </button>
    ];
  }, { strategy: 'niagara' }) {
    doSave() {
      const component = this.value();
      const nameEditor = Widget.in(this.jq().find('.componentName'));
      const fieldEditors = [ ...this.jq().find('.fe') ].map(Widget.in);

      // the field editors (with complex/slot) know how to save themselves...
      return Promise.all(fieldEditors.map((fe) => fe.save()))
        // ...and for our standalone (value) editor we must do the save logic.
        .then(() => nameEditor.read())
        .then((newName) => component.getParent().rename({
          slot: component.getName(),
          newName: baja.SlotPath.escape(newName)
        }));
    }
  }

  return PropertyTable;
});
