/*
 * A widget that demonstrates the edit, read and save features of bajaux.
 *
 * This widget also introduces the concept of using Handlebars to inject HTML
 * content for the user interface. You can find out more about Handlebars here...
 *
 * http://handlebarsjs.com/
 *
 * Note that certain module IDs include an exclamation point (!). The '!'
 * character indicates the use of a RequireJS plugin. The plugins used for this
 * module include:
 *
 * - BajaScript (baja!): ensures that BajaScript has fully started
 * - Handlebars (hbs!): import Handlebars templates
 * - Lexicon (lex!): load Lexicon data so the widget can be translated into
 *   different languages
 * - CSS (css!): import a CSS stylesheet for this widget
 */
define([
  'baja!',
  'hbs!nmodule/docDeveloper/examples/bajaux/ModifiedWidgetTemplate',
  'lex!docDeveloper',
  'bajaux/Widget',
  'bajaux/mixin/subscriberMixIn',
  'bajaux/util/SaveCommand',
  'jquery',
  'Promise' ], function (
  baja,
  template,
  lexs,
  Widget,
  subscriberMixIn,
  SaveCommand,
  $,
  Promise) {

  'use strict';

  const [ docDeveloperLex ] = lexs;

  const widgetDefaults = () => ({
    properties: { rootCssClass: 'ModifiedWidget' }
  });

  /**
   * An editor for working with `kitControl:Ramp` instances.
   *
   * @class
   * @extends module:bajaux/Widget
   * @alias module:nmodule/myFirstModule/rc/ModifiedWidget
   */
  return class ModifiedWidget extends Widget {
    constructor(params) {
      super({ params, defaults: widgetDefaults() });
      subscriberMixIn(this);

      // Add a Save Command to allow the user to save the `Widget`.
      this.getCommandGroup().add(new SaveCommand());
    }

    /**
     * Initialize the `ModifiedWidget`.
     *
     * Update the contents of the DOM in which the `Widget` is initialized. This
     * function uses the Handlebars template we imported to generate the HTML.
     *
     * This function also sets up jQuery event handlers. By default, handlers
     * registered on the `dom` parameter, like then ones we arm in this function,
     * will be automatically cleaned up when the `Widget` is destroyed. Any
     * additional handlers (on child elements of the `dom` parameter, say, or on
     * elements outside of this `Widget`) would need to be cleaned up in
     * `doDestroy()` in order to prevent memory leaks.
     *
     * @param {JQuery} dom - The DOM element into which to load this `Widget`
     */
    doInitialize(dom) {
      // The template function returns the contents of ModifiedWidget.hbs, but with
      // variables like {{valueLabel}} filled in using the properties of the object
      // argument.
      dom.html(template({
        valueLabel: docDeveloperLex.get('ModifiedWidget.value'),
        changeValuesText: docDeveloperLex.get('ModifiedWidget.changeValues'),
        amplitudeLabel: docDeveloperLex.get('ModifiedWidget.amplitude'),
        enabledLabel: docDeveloperLex.get('ModifiedWidget.enabled'),
        loadingPlaceholder: docDeveloperLex.get('ModifiedWidget.loading')
      }));

      this.getSubscriber().attach('changed', () => this.$updateDom(this.value()));

      // When the user makes a change, mark the `Widget` as modified.
      dom.on('input', '.ModifiedWidget-amplitude', () => {
        this.setModified(true);
      });

      dom.on('change', '.ModifiedWidget-enabled', () => {
        this.setModified(true);
      });
    }

    /**
     * Update the DOM to show the Ramp's current values.
     *
     * @param {baja.Component} ramp - an instance of `kitControl:Ramp`
     */
    doLoad(ramp) {
      this.$updateDom(ramp);
    }

    /**
     * Update the DOM to reflect the given Ramp's current values.
     * @private
     * @param {baja.Component} ramp a `kitControl:Ramp`
     */
    $updateDom(ramp) {
      this.$getValueDisplay().val(ramp.getOutDisplay());

      // Only update the editable DOM if the user hasn't made unsaved changes.
      if (!this.isModified()) {
        const amplitudeInput = this.$getAmplitudeInput();

        // Don't reset the user's cursor every time the value refreshes if the
        // input box has focus. They may be trying to select or edit the
        // contents.
        if (!amplitudeInput.is(':focus')) {
          amplitudeInput.val(ramp.getAmplitudeDisplay());
        }

        this.$getEnabledCheckbox().prop('checked', ramp.getEnabled());
      }
    }

    /**
     * Reads out the `enabled` and `amplitude` values that the user has currently entered.
     *
     * @returns {module:nmodule/myFirstModule/rc/ModifiedWidget~RampProperties}
     */
    doRead() {
      return {
        enabled: this.$getEnabledCheckbox().is(':checked'),
        amplitude: parseFloat(this.$getAmplitudeInput().val())
      };
    }

    /**
     * Save the user-entered changes to the loaded `kitControl:Ramp`.
     *
     * Note that the parameter to this function is the same as that resolved by
     * doRead().
     *
     * @param {module:nmodule/myFirstModule/rc/ModifiedWidget~RampProperties} readValue
     * @returns {Promise}
     */
    doSave(readValue) {
      const ramp = this.value();
      const { enabled, amplitude } = readValue;

      // Return a Promise so that the framework knows when the save has completed.
      return Promise.all([
        ramp.set({ slot: 'enabled', value: enabled }),
        ramp.set({ slot: 'amplitude', value: amplitude })
      ]);
    }

    /**
     * @private
     * @returns {JQuery} the text box for the `amplitude` property
     */
    $getAmplitudeInput() {
      return this.jq().find('.ModifiedWidget-amplitude');
    }

    /**
     * @private
     * @returns {JQuery} the checkbox for the `enabled` property
     */
    $getEnabledCheckbox() {
      return this.jq().find('.ModifiedWidget-enabled');
    }

    /**
     * @private
     * @returns {JQuery} the readonly text box showing the Ramp's value
     */
    $getValueDisplay() {
      return this.jq().find('.ModifiedWidget-value');
    }
  };

  /**
   * @typedef module:nmodule/myFirstModule/rc/ModifiedWidget~RampProperties
   * @property {boolean} enabled - whether the Ramp is enabled
   * @property {number} amplitude - the Ramp's amplitude
   */
});
