/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 */

/* eslint-env browser */

(function () {
  "use strict";

  ////////////////////////////////////////////////////////////////
  // Private vars
  ////////////////////////////////////////////////////////////////
  var matches = [],
    matchIndex = -1,
    lastDocOffset = 0,
    regex;

  ////////////////////////////////////////////////////////////////
  // Pattern matching
  ////////////////////////////////////////////////////////////////

  /**
   * Find the next occurrence of the pattern in the text from the starting point.
   */
  function findNextPatternMatch(text) {
    var match = regex.exec(text);
    if (match && match.length) {
      return {
        start: match.index,
        end: match.index + match[0].length - 1
      };
    }
    return null;
  }

  /**
   * Make a new object encapsulating the Text node from the DOM and the offsets of its
   * content within the document text.
   */
  function makeSearchNode(node, start, end) {
    return {
      docOffsetStart: start,
      docOffsetEnd: end,
      textNode: node,
      includesDocOffset: function includesDocOffset(offset) {
        return offset >= this.docOffsetStart && offset <= this.docOffsetEnd;
      }
    };
  }

  /**
   * Build an array of the Text nodes matching the search pattern, and find the offsets
   * within their text content of where the search pattern is located.
   */
  function findMatchingTextNodes(pattern, matchCase, matchWord, fromTop) {
    var text = "",
      nodes = [],
      matches = [],
      patternMatch = null,
      i = 0,
      docOffsetStart,
      docOffsetEnd,
      searchNode = 0,
      selectionStartNode,
      selectionEndNode,
      _findTextNodes,
      selectedMatchIndex;

    //
    // Find all the Text nodes in the DOM, then build a string representing the overall
    // document text content and a list of text instances representing the text node and
    // its offset into the content string.
    //

    _findTextNodes = function findTextNodes(elem) {
      var children = elem.childNodes,
        str,
        i;
      for (i = 0; i < children.length; i++) {
        if (children[i].nodeType === Node.TEXT_NODE) {
          str = children[i].nodeValue;
          nodes.push(makeSearchNode(children[i], text.length, text.length + str.length - 1));
          text = text + str;
        } else if (children[i].nodeType === Node.ELEMENT_NODE) {
          if (children[i].hasChildNodes()) {
            _findTextNodes(children[i]);
          }
        }
      }
    };
    _findTextNodes(document.body);
    while ((patternMatch = findNextPatternMatch(text)) !== null) {
      // Get the start and end offset of this instance of the pattern within the
      // whole document string.

      selectionStartNode = selectionEndNode = undefined;
      docOffsetStart = patternMatch.start;
      docOffsetEnd = patternMatch.end;

      // Find which Text node(s) contains the document string offsets. We then need to work
      // out the offset from the start of the node's text. The search pattern might span more
      // than one text node, so we need look for both the start and end nodes.

      for (i = searchNode; i < nodes.length; i++) {
        if (nodes[i].includesDocOffset(docOffsetStart)) {
          selectionStartNode = nodes[i];
        }
        if (nodes[i].includesDocOffset(docOffsetEnd)) {
          selectionEndNode = nodes[i];
        }
        if (selectionStartNode && selectionEndNode) {
          // Calculate the pattern's position in the text node's content relative to the
          // starting point of the node's text within the whole document string.

          matches.push({
            startNode: selectionStartNode.textNode,
            startNodeOffset: docOffsetStart - selectionStartNode.docOffsetStart,
            endNode: selectionEndNode.textNode,
            endNodeOffset: docOffsetEnd - selectionEndNode.docOffsetStart + 1,
            startDocOffset: docOffsetStart,
            endDocOffset: docOffsetEnd
          });

          // If not searching from the top, start by highlighting the first match
          // following the most recent selection.

          if (lastDocOffset && selectedMatchIndex === undefined) {
            if (matches[matches.length - 1].startDocOffset > lastDocOffset) {
              selectedMatchIndex = matches.length - 1;
            }
          }
          searchNode = i; // No need to look at the previous text nodes if we find another
          // occurrence of the pattern again.
          break;
        }
      }
    }
    if (matches.length) {
      matchIndex = selectedMatchIndex || 0;
      lastDocOffset = matches[matchIndex].endDocOffset;
    } else {
      lastDocOffset = 0;
      matchIndex = -1;
    }
    return matches;
  }

  /**
   * Update the browser's current text selection. This will ensure the currently selected
   * text is visible within the viewport.
   */
  function selectMatch(matches, index) {
    var selection = window.getSelection(),
      range,
      match;
    if (index >= 0 && index < matches.length) {
      match = matches[index];
      clearCurrentSelection();
      range = document.createRange();
      range.setStart(match.startNode, match.startNodeOffset);
      range.setEnd(match.endNode, match.endNodeOffset);
      selection.addRange(range);
      match.startNode.parentElement.scrollIntoView();
    }
  }

  /**
   * Remove the browser's text selection.
   */
  function clearCurrentSelection() {
    window.getSelection().removeAllRanges();
  }

  ////////////////////////////////////////////////////////////////
  // Find, find prev, find next functions
  ////////////////////////////////////////////////////////////////

  /**
   * Find the occurrences of the given pattern in the DOM's text nodes. 
   */
  function find(pattern, matchCase, matchWord, fromTop) {
    var rxpattern, flags;
    if (!pattern) {
      return false;
    }
    if (matchCase === undefined) {
      matchCase = false;
    }
    if (matchWord === undefined) {
      matchWord = false;
    }
    if (fromTop === undefined) {
      fromTop = true;
    }

    // Escape any search characters that have meaning in a regex
    rxpattern = pattern.replace(/[.*+?^${}()|[\]\\]/g, "\\$&");

    // A single space in the search pattern can match multiple whitespace chars
    // in the source text, e.g. words split by a newline char can be matched with
    // a space.
    rxpattern = rxpattern.replace(/\s+/g, "\\s+");
    if (matchWord) {
      rxpattern = "\\b" + rxpattern + "\\b";
    }
    flags = "mg";
    if (!matchCase) {
      flags = flags + "i";
    }
    regex = new RegExp(rxpattern, flags);
    if (fromTop) {
      lastDocOffset = 0;
    }
    matches = findMatchingTextNodes(pattern, matchCase, matchWord, fromTop);
    if (matches.length) {
      selectMatch(matches, matchIndex);
      return true;
    }
    clearCurrentSelection();
    return false;
  }

  /**
   * Find the next occurrence of the searched for text. Called
   * after find(). Return true if another occurrence is found, false if
   * there are no more.
   */
  function findNext() {
    if (matches && matches.length && matchIndex + 1 < matches.length) {
      matchIndex = matchIndex + 1;
      lastDocOffset = matches[matchIndex].endDocOffset;
      selectMatch(matches, matchIndex);
      return true;
    }
    return false;
  }

  /**
   * Find the previous occurrence of the searched for text. Called
   * after find(). Return true if another occurrence is found, false if
   * there are no more.
   */
  function findPrev() {
    if (matches && matches.length && matchIndex - 1 >= 0) {
      matchIndex = matchIndex - 1;
      lastDocOffset = matches[matchIndex].endDocOffset;
      selectMatch(matches, matchIndex);
      return true;
    }
    return false;
  }

  //
  // Set the public functions on the window.niagara.findtext object.
  // These are intended to be called from the BWebBrowserView toolbar
  // commands.
  //

  window.niagara = window.niagara || {};
  window.niagara.findtext = window.niagara.findtext || {
    find: find,
    findNext: findNext,
    findPrev: findPrev
  };
  return window.niagara.findtext;
})();
//# sourceMappingURL=findCommands.js.map
