/**
 * @license Copyright 2016, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview MomentaryToggleBinding class
 * @author Danesh Kamal
 */

/**
 * @private
 * @module mobile/px/bindings/MomentaryToggleBinding
 */
define(['baja!', 'baja!baja:Boolean,baja:StatusBoolean,control:BooleanWritable', 'jquery', 'mobile/px/bindings/Binding', 'Promise'], function (baja, types, $, Binding, Promise) {
  "use strict";

  /**
   * A Component that represents a Niagara 'kitPx:MomentaryToggleBinding' Type.
   *
   * @private
   * @class
   * @alias module:mobile/px/bindings/MomentaryToggleBinding
   * @extends mobile/px/bindings/Binding
   */
  var MomentaryToggleBinding = function MomentaryToggleBinding() {
    Binding.apply(this, arguments);
  };
  MomentaryToggleBinding.prototype = Object.create(Binding.prototype);
  MomentaryToggleBinding.prototype.constructor = MomentaryToggleBinding;

  /**
   * Sets the boolean state on the binding target based on whether the target is
   * a baja:Boolean, an Action with a baja:Boolean parameter or a baja:StatusBoolean
   *
   * @private
   * @param {boolean} value
   * @returns {Promise} Promise resolved when target state has been set to boolean state
   * of the widget or Boolean indicating value could not be set.
   */
  MomentaryToggleBinding.prototype.$setValueOnTarget = function (value) {
    var target = this.$target,
      targetPressed = this.$targetPressed,
      component = target.getComponent(),
      slot = target.slot;
    if (!value && !targetPressed) {
      return Promise.resolve();
    }
    this.$targetPressed = value;
    if (slot) {
      if (slot.isProperty()) {
        var typeSpec = slot.getType().getTypeSpec();
        if (typeSpec === 'baja:Boolean') {
          return component.set({
            slot: slot,
            value: value
          });
        } else if (typeSpec === 'baja:StatusBoolean') {
          var statusBoolean = component.get(slot);
          return Promise.all([statusBoolean.set({
            slot: 'status',
            value: baja.Status.make({
              orig: statusBoolean.getStatus(),
              bits: baja.Status.NULL,
              state: false
            })
          }), statusBoolean.set({
            slot: 'value',
            value: value
          })]);
        } else {
          throw new Error('configuration error, property must be Boolean or StatusBoolean');
        }
      } else if (slot.isAction()) {
        var paramType = slot.getParamType();
        if (paramType && paramType.getTypeSpec() === 'baja:Boolean') {
          return component.invoke({
            slot: slot,
            value: value
          });
        } else {
          throw new Error('configuration error, action must take boolean parameter');
        }
      } else {
        throw new Error('configuration error, slot must be property or action');
      }
    } else {
      if (baja.hasType(component, 'control:BooleanWritable')) {
        return component.invoke({
          slot: 'set',
          value: value
        });
      } else {
        throw new Error('configuration error, target must be BooleanWritable');
      }
    }
  };
  MomentaryToggleBinding.prototype.$isButtonPressed = function () {
    return this.getWidget().isPressed();
  };

  /**
   * Called when the binding is updated.
   */
  MomentaryToggleBinding.prototype.update = function () {
    var _this = this,
      _arguments = arguments;
    return Promise["try"](function () {
      return _this.$setValueOnTarget(_this.$isButtonPressed());
    }).then(function () {
      return Binding.prototype.update.apply(_this, _arguments);
    });
  };

  /**
   * Handle an event from the Widget.
   *
   * @return {Boolean} return true if this binding has handled the event.
   */
  MomentaryToggleBinding.prototype.handleEvent = function () {
    return !!this.$setValueOnTarget(this.$isButtonPressed());
  };
  return MomentaryToggleBinding;
});
