/**
 * @license Copyright 2012, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview PxPage class
 *
 * @author Gareth Johnson
 * @version 0.0.2.0
 */
define(['baja!', 'baja!bajaui:RootContainer', 'lex!mobile', 'bajaux/events', 'jquery', 'jquerymobile', 'Promise', 'underscore', 'mobile/px/bindings/registerbindings', 'mobile/px/converters/registerconverters', 'mobile/px/gx/registergx', 'mobile/px/widgets/registerwidgets', 'mobile/util/mobile/dialogs', 'mobile/util/mobile/pages'], function (baja, types, lexs, events, $, jqm, Promise, _, registerBindings, registerConverters, registerGx, registerWidgets, dialogs, pages) {
  "use strict";

  var isPreviousContentMobilePane = true,
    mobileLex = lexs[0],
    MODIFY_EVENT = events.MODIFY_EVENT,
    SAVE_EVENT = events.SAVE_EVENT;
  registerBindings = _.once(registerBindings);
  registerConverters = _.once(registerConverters);
  registerGx = _.once(registerGx);
  registerWidgets = _.once(registerWidgets);

  /**
   * Called if the Px page fails to load.
   *
   * @param err
   */
  function pxFailed(err) {
    dialogs.error(err);
  }

  /**
   * Update the bindings from the result of the batch request.
   *
   * @param pxPage
   * @param {baja.BatchResolve} batchResolve
   */
  function updateBindings(pxPage, batchResolve) {
    var target,
      navOrd,
      comp,
      i,
      boundBindings = pxPage.$boundBindings,
      byComponent = pxPage.$byComponent,
      allBindings = pxPage.$allBindings;

    // Detach all current event handlers
    pxPage.$subscriber.detach();

    // Attach the changed handler now that everything is subscribed
    pxPage.$subscriber.attach("changed", function () {
      var navOrd = this.getNavOrd();
      if (pxPage.$byComponent.hasOwnProperty(navOrd)) {
        var b = pxPage.$byComponent[navOrd],
          i;
        for (i = 0; i < b.length; ++i) {
          b[i].update();
        }
      }
    });

    // Set the ORD Targets up in the binding Components
    for (i = 0; i < batchResolve.size(); ++i) {
      target = null;
      if (batchResolve.isResolved(i)) {
        target = batchResolve.getTarget(i);
        boundBindings[i].$fw("updateBinding", target);
        comp = target.getComponent();
        if (comp && comp.isMounted()) {
          navOrd = comp.getNavOrd();
          if (!byComponent.hasOwnProperty(navOrd)) {
            byComponent[navOrd] = [boundBindings[i]];
          } else {
            byComponent[navOrd].push(boundBindings[i]);
          }
        }
      } else {
        baja.error("Failed to resolve: " + boundBindings[i].getOrd().toString());
      }
    }

    // Update all bindings
    var loads = _.map(allBindings, function (binding) {
      return binding.load().then(function () {
        return binding.update(/*firstUpdate*/true);
      });
    });
    return Promise.all(loads).then(function () {
      // show page content
      pxPage.$rootDom.show();

      // Update the commands button depending on whether the Px page is modified
      $("a.commandsButton", pxPage.$headerDom.parent()).toggleClass("red", pxPage.$rootWidget.isWidgetTreeModified());

      // Update the header with the display name
      pxPage.$headerDom.text(pxPage.$displayName);

      // Update the main HTML title text...
      try {
        document.title = pxPage.$displayName;
      } catch (ignore) {
        // Believe it or not, IE doesn't seem to like changing the title text so we need to catch any errors here
      }
      pxPage.$rootWidget.getDomElement().trigger("pxPageLoaded");
    });
  }

  /**
   * Recursively search for all of the bindings.
   *
   * @param pxPage
   * @param comp the Component to search for bindings on.
   */
  function findBindings(pxPage, comp) {
    if (!comp) {
      return;
    }

    // Make a note of any bindings we find
    if (comp.getType().is("bajaui:Binding")) {
      if (comp.getOrd() !== baja.Ord.DEFAULT) {
        // Join base ORD and Binding ORD so BatchResolve will normalize and process a complete absolute ORD
        pxPage.$ords.push(baja.Ord.make({
          base: pxPage.$viewOrd,
          child: comp.getOrd().toString()
        }));
        pxPage.$boundBindings.push(comp);
      }
      pxPage.$allBindings.push(comp);
    }
    var map = comp.$map.$map,
      // Use internal OrderedMap for max efficiency
      p;

    // Search all Widgets for bindings      
    for (p in map) {
      if (map.hasOwnProperty(p) && map[p].isProperty() && map[p].getType().isComponent()) {
        findBindings(pxPage, comp.get(map[p]));
      }
    }
  }
  function postLoadWidgetsHasUpdate() {
    return true;
  }

  /**
   * Recursively called once all Widgets have been loaded so they can update for the first time.
   *
   * @param pxPage
   * @param widget
   */
  function postLoadWidgets(pxPage, widget) {
    function loadKids() {
      var map = widget.$map.$map,
        // Use internal OrderedMap for max efficiency
        kidWidgets = _.chain(map).filter(function (slot) {
          return slot.isProperty() && slot.getType().is('bajaui:Widget');
        }).map(function (slot) {
          return widget.get(slot);
        }).value(),
        loads = _.map(kidWidgets, function (kid) {
          return postLoadWidgets(pxPage, kid);
        });
      return Promise.all(loads);
    }

    // Check we have the correct methods defined on this objects
    if (typeof widget.isModified === "function" && typeof widget.save === "function") {
      pxPage.$rootWidget.addSavableWidget(widget);
    }

    // Internally set the root Widget for performance
    widget.$root = pxPage.$rootWidget;

    // Only post load and update if there's a valid DOM element available for the Widget
    if (widget.getDomElement()) {
      // Post load the Widget after the page has been enhanced
      widget.postLoad();

      // First update
      return widget.update(postLoadWidgetsHasUpdate, /*firstUpdate*/true).then(loadKids);
    } else {
      return loadKids();
    }
  }

  /**
   * Load the Widgets for the Px page
   *
   * @param pxPage
   */
  function loadWidgets(pxPage) {
    // because other modules have implemented some of these as type extensions,
    // we do a lazy type registration just before instantiation to ensure that
    // the old/existing mobile versions *overwrite* those and continue to be used.
    registerConverters();
    registerBindings();
    registerGx();
    registerWidgets();

    // Load the Widget tree and set a root container
    var rootWidget = pxPage.$rootWidget = baja.$("bajaui:RootContainer");
    rootWidget.$pxPage = pxPage;
    rootWidget.add({
      slot: "content",
      value: baja.bson.decodeValue(pxPage.$bson, baja.$serverDecodeContext)
    });

    // Create the room DOM element
    var rootDom = pxPage.$rootDom = $("<div></div>");

    // Create DOM structure
    rootWidget.load(rootDom);
    rootWidget.loadChildren();

    // Now all the content is added, dynamically update the page.

    // first, hide the element so we don't get reflows during JQM enhancement.
    rootDom.hide();
    // attach to document so JQM will see the 'create' event
    pxPage.$contentDom.append(rootDom);
    // do JQM enhancement...
    rootDom.enhanceWithin();

    // now continue with widget loading and updating with root dom hidden
    // to minimize repaints. will be shown when appropriate in
    // updateBindings.

    // Post load and update the Widgets and find bindings
    return Promise.join(postLoadWidgets(pxPage, rootWidget), findBindings(pxPage, pxPage.$rootWidget));
  }

  /**
   * @class PxPage
   *
   * Represents a Px page.
   *
   * @param pageDiv the page DOM element
   * @param bson the BSON encoding of a Px page that we need to load
   * @param {baja.Ord} viewOrd the view ORD
   * @param {String} displayName the display name of the Px view
   */
  function PxPage(pageDiv, bson, viewOrd, displayName) {
    this.$allBindings = []; // All the bindings found
    this.$boundBindings = []; // All the bindings with ORDs to be bound
    this.$ords = []; // All of the ORDs to be resolved for the page
    this.$byComponent = {}; // A Component lookup map to associate Components to Bindings
    this.$subscriber = new baja.Subscriber();
    this.$contentDom = pageDiv.find(".pxContent");
    this.$headerDom = pageDiv.find(".pxHeader");
    this.$bson = bson;
    this.$started = false;
    this.$viewOrd = viewOrd;
    this.$displayName = displayName;
    this.$rootWidget = null;
  }

  /**
   * Start the Px page.
   * 
   * This will subscribe any valid bindings on the Px page.
   */
  PxPage.prototype.start = function () {
    var that = this,
      contentDom = that.$contentDom;

    // If already started then don't stop
    if (that.$started) {
      return;
    }
    that.$started = true;

    // Update the commands button if the page is modified
    contentDom.on(MODIFY_EVENT + ' ' + SAVE_EVENT, that.$modifyHandler = baja.throttle(function () {
      $("a.commandsButton", that.$headerDom.parent()).toggleClass("red", PxPage.isCurrentPxPageModified());
    }, 500));

    // TODO: Consider any timing issues between subscription and unsubscription on start and stop.

    // Resolve the bindings and Subscribe too all mounted Components
    var batchResolve = new baja.BatchResolve(that.$ords),
      isContentMobilePane = that.$rootWidget.get("content").getType().is("mobile:IMobilePane");

    // Keep track of previous state to avoid any unnecessary reflows...      
    if (isPreviousContentMobilePane !== isContentMobilePane) {
      isPreviousContentMobilePane = isContentMobilePane;
      $(".ui-mobile-viewport,.ui-content").toggleClass("nonRootMobilePane", !isContentMobilePane);
      $("div[data-role='page']").toggleClass("nonRootMobilePanePage", !isContentMobilePane);
    }
    return batchResolve.resolve({
      subscriber: that.$subscriber
    })
    // eslint-disable-next-line promise/no-return-in-finally
    ["finally"](function () {
      return updateBindings(that, batchResolve).then(function () {
        that.$rootWidget.getDomElement().trigger("pxPageStarted");
      });
    });
  };

  /**
   * Stop the Px page.
   * 
   * This will unsubscribe any bound bindings on the Px page.
   */
  PxPage.prototype.stop = function () {
    var that = this,
      modifyHandler = that.$modifyHandler;

    // Set the header to loading
    that.$headerDom.text(mobileLex.get("loading"));
    if (modifyHandler) {
      that.$contentDom.off(MODIFY_EVENT + ' ' + SAVE_EVENT, modifyHandler);
    }
    if (that.$started) {
      that.$started = false;
      that.$subscriber.detach();
      that.$subscriber.unsubscribeAll({
        ok: function ok() {
          that.$rootWidget.getDomElement().trigger("pxPageStopped");
        },
        fail: baja.fail
      });
    }
  };

  /**
   * Load the Px page.
   * 
   * This will decode and load all of the Widgets and Bindings for the Px page.
   */
  PxPage.prototype.load = function () {
    var that = this,
      loaded = that.$loaded,
      bson = that.$bson;
    if (loaded) {
      return Promise.resolve();
    }

    // Scan BSON for unknown Types and make a network call if necessary
    // eslint-disable-next-line promise/avoid-new
    return new Promise(function (resolve, reject) {
      function importOk() {
        try {
          // Decode the Px BSON embedded into the page
          that.$contentDom.empty();
          loadWidgets(that).then(function () {
            return that.start();
          }).then(function () {
            that.$loaded = true;
            resolve();
          })["catch"](reject);
        } catch (err) {
          reject(err);
        }
      }
      baja.bson.importUnknownTypes(bson, importOk, pxFailed);
    });
  };

  /**
   * Return the ORD for this view.
   * 
   * @returns {baja.Ord}
   */
  PxPage.prototype.getViewOrd = function () {
    return this.$viewOrd;
  };

  /**
   * Return the display name for the view.
   * 
   * @returns {String}
   */
  PxPage.prototype.getDisplayName = function () {
    return this.$displayName;
  };

  /**
   * Return true if the current Px Page is modified.
   *
   * @returns {Boolean}
   */
  PxPage.isCurrentPxPageModified = function () {
    var pageData = pages.getCurrent().jqmData("pageData");
    if (pageData && pageData.pxPage && pageData.pxPage.$rootWidget) {
      return pageData.pxPage.$rootWidget.isWidgetTreeModified();
    } else {
      return false;
    }
  };

  /**
   * Return the current Px Page (or null if there isn't one available)
   *
   * @returns PxPage (or null if there is none available).
   */
  PxPage.getCurrentPxPage = function () {
    var pageData = pages.getCurrent().jqmData("pageData");
    return pageData && pageData.pxPage ? pageData.pxPage : null;
  };
  return PxPage;
});
