/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/schedule/rc/fe/ScheduleBlockEditor
 */
define(['Promise', 'nmodule/gx/rc/util/colorUtils', 'nmodule/schedule/rc/model/ScheduleBlock', 'nmodule/schedule/rc/model/TimeOfDay', 'nmodule/webEditors/rc/fe/BaseWidget'], function (Promise, colorUtils, ScheduleBlock, TimeOfDay, BaseWidget) {
  'use strict';

  var getContrastingColor = colorUtils.getContrastingColor,
    parseRgba = colorUtils.parseRgba,
    toRgbCssString = colorUtils.toRgbCssString;
  var tpl = function tpl(ed) {
    return "\n    <div class=\"grab top\"/>\n    <div class=\"grab bottom\"/>\n    <div class=\"scheduleBlock\">\n      <div class=\"display\">\n        <span class=\"valueDisplay\"></span><span class=\"timeDisplay\"></span>\n      </div>\n    </div>";
  };

  /**
   * `ScheduleBlockEditor` represents a single block in the schedule for one
   * day.
   *
   * It supports the following bajaux `Properties`:
   *
   * - `background`: {string} a CSS color for the block background
   * - `foreground`: {string} a CSS color for the block text color; or, the
   *   special value "contrast" to calculate a text color that contrasts
   *   legibly with the background
   *
   * @class
   * @alias module:nmodule/schedule/rc/fe/ScheduleBlockEditor
   * @extends module:nmodule/webEditors/rc/fe/BaseWidget
   */
  var ScheduleBlockEditor = function ScheduleBlockEditor() {
    BaseWidget.apply(this, arguments);
  };
  ScheduleBlockEditor.prototype = Object.create(BaseWidget.prototype);
  ScheduleBlockEditor.prototype.constructor = ScheduleBlockEditor;
  ScheduleBlockEditor.prototype.doChanged = function (name, value) {
    switch (name) {
      case 'background':
      case 'foreground':
        return this.$updateColors();
    }
  };
  /**
   * @param {jQuery} dom
   */
  ScheduleBlockEditor.prototype.doInitialize = function (dom) {
    dom.addClass('ScheduleBlockEditor').html(tpl(this));
    this.$updateColors();
  };

  /**
   * @param {module:nmodule/schedule/rc/model/ScheduleBlock} block
   */
  ScheduleBlockEditor.prototype.doLoad = function (block) {
    this.$getTimeDisplayElement().text(block.getTimeRangeDisplay());
    this.$getValueDisplayElement().text(block.getValueDisplay());
  };
  ScheduleBlockEditor.prototype.doDestroy = function () {
    this.jq().removeClass('ScheduleBlockEditor');
  };
  function isTimeOfDay(obj) {
    return obj instanceof TimeOfDay;
  }

  /**
   * Set the desired start and finish time at once, and validate to ensure that
   * the start/finish range is valid. `this.value()` will return a new
   * `ScheduleBlock` instance afterward. If the range changes,
   * `this.isModified()` will then return true.
   *
   * @param {module:nmodule/schedule/rc/model/TimeOfDay} start
   * @param {module:nmodule/schedule/rc/model/TimeOfDay} finish
   * @param {string} [timeRangeDisplay]
   * @returns {Promise}
   */
  ScheduleBlockEditor.prototype.setTimeRange = function (start, finish, timeRangeDisplay) {
    if (!isTimeOfDay(start) || !isTimeOfDay(finish)) {
      return Promise.reject(new Error('TimeOfDay start and finish required'));
    }
    var startMs = start.getTimeOfDayMillis(),
      finishMs = finish.getTimeOfDayMillis(),
      value = this.value();
    if (finishMs > 0 && startMs >= finishMs) {
      return Promise.reject(new Error('start must be before finish'));
    }
    if (!start.equals(value.getStart()) || !finish.equals(value.getFinish())) {
      this.setModified(true);
    }
    return this.load(new ScheduleBlock({
      start: start,
      finish: finish,
      timeRangeDisplay: timeRangeDisplay,
      value: value.getValue(),
      valueDisplay: value.getValueDisplay()
    }));
  };
  ScheduleBlockEditor.prototype.$getTimeDisplayElement = function () {
    return this.jq().find('.timeDisplay');
  };
  ScheduleBlockEditor.prototype.$getValueDisplayElement = function () {
    return this.jq().find('.valueDisplay');
  };
  ScheduleBlockEditor.prototype.$updateColors = function () {
    var props = this.properties(),
      background = props.getValue('background') || '',
      color = props.getValue('foreground') || '';
    if (color === 'contrast' && background) {
      color = toRgbCssString(getContrastingColor(parseRgba(background)));
    }
    this.jq().children('.scheduleBlock').css({
      background: background,
      color: color
    });
  };
  return ScheduleBlockEditor;
});
