/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/schedule/rc/fe/DayEditor
 */
define(['lex!baja', 'bajaux/commands/Command', 'bajaux/events', 'jquery', 'Promise', 'underscore', 'nmodule/schedule/rc/fe/ScheduleBlockEditor', 'nmodule/schedule/rc/fe/mixin/DayEditorDragSupport', 'nmodule/schedule/rc/model/DaySchedule', 'nmodule/schedule/rc/model/ScheduleBlock', 'nmodule/schedule/rc/model/TimeOfDay', 'nmodule/schedule/rc/util/scheduleUtils', 'nmodule/webEditors/rc/fe/CompositeEditor', 'nmodule/webEditors/rc/fe/config/CompositeBuilder'], function (lexs, Command, events, $, Promise, _, ScheduleBlockEditor, addDragSupport, DaySchedule, ScheduleBlock, TimeOfDay, scheduleUtils, CompositeEditor, CompositeBuilder) {
  'use strict';

  var bajaLex = lexs[0],
    LOAD_EVENT = events.LOAD_EVENT,
    MODIFY_EVENT = events.MODIFY_EVENT,
    setTopAndBottom = scheduleUtils.setTopAndBottom;
  var toBlock = function toBlock(i) {
    return "<div class=\"block block".concat(i, "\" />");
  };
  var tpl = function tpl(ed) {
    return "\n    <div class=\"day\">\n      <div class=\"dayTitle\"></div>\n      <div class=\"blocksContainer\">\n        <div class=\"blocksDisplay\">".concat(_.range(0, 8).map(toBlock).join('\n'), "</div>\n      </div>\n    </div>");
  };

  ////////////////////////////////////////////////////////////////
  // Commands
  ////////////////////////////////////////////////////////////////

  var AllDayCommand = function AllDayCommand(ed) {
    Command.call(this, {
      module: 'schedule',
      lex: 'commands.allDay',
      func: function func() {
        var block = ed.$getSelectedBlock(),
          value = block && block.getValue();
        return ed.createScheduleBlock(new TimeOfDay(0), new TimeOfDay(0), value).then(function (allDayBlock) {
          return ed.$getApplyCommand().invoke([allDayBlock]);
        }).then(function () {
          var block = ed.$getScheduleBlocks()[0];
          ed.$setSelectedBlock(block);
          ed.trigger('selectionChange', block);
        });
      }
    });
  };
  AllDayCommand.prototype = Object.create(Command.prototype);
  AllDayCommand.prototype.constructor = AllDayCommand;
  var ApplyCommand = function ApplyCommand(ed) {
    Command.call(this, {
      func: function func(blocks) {
        return ed.load(new DaySchedule({
          dayOfWeek: ed.value().getDayOfWeek(),
          blocks: blocks
        })).then(function () {
          return ed.setModified(true);
        });
      }
    });
  };
  ApplyCommand.prototype = Object.create(Command.prototype);
  ApplyCommand.prototype.constructor = ApplyCommand;
  var ClearCommand = function ClearCommand(ed) {
    Command.call(this, {
      module: 'schedule',
      lex: 'commands.clearDay',
      func: function func() {
        return ed.$getApplyCommand().invoke([]);
      }
    });
  };
  ClearCommand.prototype = Object.create(Command.prototype);
  ClearCommand.prototype.constructor = ClearCommand;
  var DeleteCommand = function DeleteCommand(ed) {
    Command.call(this, {
      module: 'schedule',
      lex: 'commands.deleteEvent',
      func: function func() {
        var selectedBlock = ed.$getSelectedBlock();
        if (selectedBlock) {
          return ed.deleteScheduleBlock(selectedBlock);
        }
      }
    });
  };
  DeleteCommand.prototype = Object.create(Command.prototype);
  DeleteCommand.prototype.constructor = DeleteCommand;
  var ScheduleDefaultsCommand = function ScheduleDefaultsCommand(ed) {
    Command.call(this, {
      module: 'schedule',
      lex: 'commands.scheduleDefaults',
      func: function func() {
        return ed.scheduleDefaults();
      }
    });
  };
  ScheduleDefaultsCommand.prototype = Object.create(Command.prototype);
  ScheduleDefaultsCommand.prototype.constructor = ScheduleDefaultsCommand;
  var EditCommand = function EditCommand(ed) {
    Command.call(this, {
      module: 'schedule',
      lex: 'commands.editEvent',
      func: function func() {
        var selectedBlockEditor = ed.$getSelectedBlockEditor();
        if (selectedBlockEditor) {
          return ed.editScheduleBlock(selectedBlockEditor);
        }
      }
    });
  };
  EditCommand.prototype = Object.create(Command.prototype);
  EditCommand.prototype.constructor = EditCommand;

  ////////////////////////////////////////////////////////////////
  // Definition
  ////////////////////////////////////////////////////////////////

  /**
   * `DayEditor` is a composite editor that handles the "draggy bars"
   * functionality for one day in a scheduled week. Each block within the day's
   * schedule will be available as a block that can be moved and resized.
   *
   * @class
   * @alias module:nmodule/schedule/rc/fe/DayEditor
   * @extends module:nmodule/webEditors/rc/fe/CompositeEditor
   */
  var DayEditor = function DayEditor() {
    CompositeEditor.apply(this, arguments);
    addDragSupport(this);
    if (!this.properties().get('effectiveValue')) {
      this.properties().add({
        name: 'effectiveValue',
        value: null,
        "transient": true,
        hidden: true
      });
    }
    this.getCommandGroup().add(new AllDayCommand(this), new ApplyCommand(this), new ClearCommand(this), new DeleteCommand(this), new ScheduleDefaultsCommand(this), new EditCommand(this));
  };
  DayEditor.prototype = Object.create(CompositeEditor.prototype);
  DayEditor.prototype.constructor = DayEditor;
  DayEditor.prototype.$getAllDayCommand = function () {
    return this.getCommandGroup().get(0);
  };
  DayEditor.prototype.$getApplyCommand = function () {
    return this.getCommandGroup().get(1);
  };
  DayEditor.prototype.$getClearCommand = function () {
    return this.getCommandGroup().get(2);
  };
  DayEditor.prototype.$getDeleteCommand = function () {
    return this.getCommandGroup().get(3);
  };
  DayEditor.prototype.$getScheduleDefaultsCommand = function () {
    return this.getCommandGroup().get(4);
  };
  DayEditor.prototype.$getEditCommand = function () {
    return this.getCommandGroup().get(5);
  };

  /**
   * The builder for a `DayEditor` maps the schedule blocks contained within
   * this day to individual draggable editors.
   *
   * @returns {module:nmodule/webEditors/rc/fe/config/CompositeBuilder}
   */
  DayEditor.prototype.makeBuilder = function () {
    var that = this,
      builder = new CompositeBuilder();
    builder.getKeys = function () {
      var daySchedule = this.getDataSource();
      return _.compact(daySchedule.getBlocks().map(function (block, i) {
        return block && String(i);
      }));
    };
    builder.getValueFor = function (key) {
      var daySchedule = this.getDataSource();
      return daySchedule.getBlocks()[key];
    };
    builder.getConfigFor = function () {
      return {
        type: ScheduleBlockEditor
      };
    };
    builder.getDomFor = function (key) {
      var block = this.getValueFor(key);
      return setTopAndBottom($('<div/>'), block).appendTo(that.$getBlocksDisplayElement());
    };
    return builder;
  };

  ////////////////////////////////////////////////////////////////
  // Edit functions
  ////////////////////////////////////////////////////////////////

  /**
   * Add a new schedule block to the day.
   * @param {module:nmodule/schedule/rc/model/ScheduleBlock} scheduleBlock
   * @returns {Promise} promise to be resolved after the block has been added
   * and its corresponding editor has been built
   */
  DayEditor.prototype.addScheduleBlock = function (scheduleBlock) {
    var that = this;
    if (!(that.value() && !that.$loading)) {
      return reject('must load a DaySchedule first');
    }
    if (!(scheduleBlock instanceof ScheduleBlock)) {
      return reject('ScheduleBlock required');
    }
    var builder = that.getBuilder(),
      value = that.value(),
      blocks = value.getBlocks(),
      newKey = String(blocks.length);
    blocks.push(scheduleBlock);
    return builder.$loadFor(newKey).then(function () {
      that.setModified(true);
      that.$setDefaultEffectiveValue(scheduleBlock.getValue());
      return builder.getEditorFor(newKey);
    });
  };

  /**
   * Check to see if the given block editor can move to the specified time range
   * without overlapping any other blocks in the `DayEditor`.
   *
   * @param {module:nmodule/schedule/rc/fe/ScheduleBlockEditor} blockEditor
   * @param {module:nmodule/schedule/rc/model/TimeOfDay} start
   * @param {module:nmodule/schedule/rc/model/TimeOfDay} finish
   * @returns {boolean}
   */
  DayEditor.prototype.canMoveBlockTo = function (blockEditor, start, finish) {
    var blocks = this.$getScheduleBlocks(),
      myBlockEditor,
      myBlock;
    for (var i = 0, len = blocks.length; i < len; i++) {
      myBlockEditor = blocks[i];
      myBlock = myBlockEditor.value();
      if (myBlockEditor !== blockEditor && myBlock.overlaps(start, finish)) {
        return false;
      }
    }
    return true;
  };

  /**
   * Create a new schedule block.
   * @param {module:nmodule/schedule/rc/model/TimeOfDay}start
   * @param {module:nmodule/schedule/rc/model/TimeOfDay}finish
   * @param {*} [value] specify a value for the block - otherwise the current
   * value of the `effectiveValue` Property will be used
   * @returns {Promise.<module:nmodule/schedule/rc/model/ScheduleBlock>}
   */
  DayEditor.prototype.createScheduleBlock = function (start, finish, value) {
    if (!start) {
      return reject('start required');
    }
    if (!finish) {
      return reject('finish required');
    }
    if (typeof value === 'undefined') {
      value = this.$getDefaultEffectiveValue();
    }
    return Promise.resolve(new ScheduleBlock({
      start: start,
      finish: finish,
      value: value
    }));
  };

  /**
   * Remove the specified block from this day.
   * @param {module:nmodule/schedule/rc/model/ScheduleBlock} scheduleBlock
   * @returns {Promise} promise to be resolved once the block has been removed
   * and its corresponding editor has been destroyed
   */
  DayEditor.prototype.deleteScheduleBlock = function (scheduleBlock) {
    var that = this,
      blocks = that.value().getBlocks(),
      blockEditors = that.$getScheduleBlocks();
    _.find(blockEditors, function (blockEditor) {
      if (scheduleBlock === blockEditor.value()) {
        delete blocks[blockEditor.jq().data('key')];
        that.setModified(true);
        return true;
      }
    });
    return that.getBuilder().$prune();
  };

  /**
   * Allow the user to make changes to the selected block editor. Because the
   * editor can any form, override this function as needed for your use case.
   * By default, does nothing.
   * @param {module:nmodule/schedule/rc/fe/ScheduleBlockEditor} blockEditor
   * @returns {Promise}
   */
  DayEditor.prototype.editScheduleBlock = function (blockEditor) {
    return Promise.resolve();
  };

  /**
   * Fill the empty portions of the day with property.effectiveValue.
   *
   * @returns {Promise}
   */
  DayEditor.prototype.scheduleDefaults = function () {
    var that = this,
      value = that.$getDefaultOutput();
    return that.readScheduleBlocks().then(function (scheduleBlocks) {
      var creationPromises = [];
      // If empty, just create a single all day event
      if (scheduleBlocks.length === 0) {
        creationPromises.push(that.createScheduleBlock(new TimeOfDay(0), new TimeOfDay(0), value));
      } else {
        // Sort by getStart().getTimeOfDayMillis()
        scheduleBlocks = scheduleBlocks.sort(function (a, b) {
          return a.getStart().getTimeOfDayMillis() - b.getStart().getTimeOfDayMillis();
        });

        // Iterate through the blocks and create new interleaving blocks
        var startTime = 0,
          endTime;
        scheduleBlocks.forEach(function (scheduleBlock) {
          var nextStart = scheduleBlock.getStart().getTimeOfDayMillis();
          if (startTime < nextStart) {
            creationPromises.push(that.createScheduleBlock(newTime(startTime), newTime(nextStart), value));
          }
          startTime = endTime = scheduleBlock.getFinish().getTimeOfDayMillis();
          creationPromises.push(scheduleBlock);
        });
        if (endTime) {
          creationPromises.push(that.createScheduleBlock(newTime(endTime), newTime(0), value));
        }
      }
      return Promise.all(creationPromises);
    }).then(function (createdBlocks) {
      return that.$getApplyCommand().invoke(createdBlocks);
    }).then(function () {
      var block = that.$getScheduleBlocks()[0];
      that.$setSelectedBlock(block);
      that.trigger('selectionChange', block);
    });
  };

  /**
   * Read the current values of all schedule blocks.
   * @returns {Promise.<Array.<module:nmodule/schedule/rc/model/ScheduleBlock>>}
   */
  DayEditor.prototype.readScheduleBlocks = function () {
    return this.$getScheduleBlocks().readAll();
  };

  ////////////////////////////////////////////////////////////////
  // bajaux implementation
  ////////////////////////////////////////////////////////////////

  /**
   * @param {jQuery} dom
   */
  DayEditor.prototype.doInitialize = function (dom) {
    dom.addClass('DayEditor').html(tpl(this));

    //when a block is created or changed, ensure its height is correctly set
    dom.on([LOAD_EVENT, MODIFY_EVENT].join(' '), '.ScheduleBlockEditor', function (e, ed) {
      setTopAndBottom($(this), ed.value());
    });
  };

  /**
   * @param {module:nmodule/schedule/rc/model/DaySchedule} daySchedule
   * @returns {Promise}
   */
  DayEditor.prototype.doLoad = function (daySchedule) {
    this.$getDayNameElement().text(bajaLex.get(daySchedule.getDayOfWeek() + '.short'));
    return CompositeEditor.prototype.doLoad.apply(this, arguments);
  };

  /**
   * @returns {Promise.<module:nmodule/schedule/rc/model/DaySchedule>}
   */
  DayEditor.prototype.doRead = function () {
    var dayOfWeek = this.value().getDayOfWeek();
    return this.getChildWidgets().readAll().then(function (blocks) {
      return new DaySchedule({
        dayOfWeek: dayOfWeek,
        blocks: blocks
      });
    });
  };
  DayEditor.prototype.doReadonly = function (readonly) {
    scheduleUtils.configureEditCommandReadonly(this.$getEditCommand(), !this.isEnabled() || readonly);
    return CompositeEditor.prototype.doReadonly.apply(this, arguments);
  };
  DayEditor.prototype.doEnabled = function (enabled) {
    scheduleUtils.configureEditCommandReadonly(this.$getEditCommand(), !enabled || this.isReadonly());
    return CompositeEditor.prototype.doEnabled.apply(this, arguments);
  };

  /**
   * @returns {Promise}
   */
  DayEditor.prototype.doDestroy = function () {
    this.jq().removeClass('DayEditor');
    return CompositeEditor.prototype.doDestroy.apply(this, arguments);
  };

  ////////////////////////////////////////////////////////////////
  // Support functions
  ////////////////////////////////////////////////////////////////

  /**
   * @private
   * @returns {jQuery}
   */
  DayEditor.prototype.$getBlocksContainerElement = function () {
    return this.jq().children('.day').children('.blocksContainer');
  };

  /**
   * @private
   * @returns {jQuery}
   */
  DayEditor.prototype.$getBlocksDisplayElement = function () {
    return this.$getBlocksContainerElement().children('.blocksDisplay');
  };

  /**
   * @private
   * @returns {jQuery}
   */
  DayEditor.prototype.$getDayNameElement = function () {
    return this.jq().children('.day').children('.dayTitle');
  };

  /**
   * Get the configured value intended for use in constructing new schedule
   * blocks.
   * @returns {*}
   */
  DayEditor.prototype.$getDefaultEffectiveValue = function () {
    return this.properties().getValue('effectiveValue');
  };
  DayEditor.prototype.$getDefaultOutput = function () {
    return this.properties().getValue('defaultOutput');
  };

  /**
   * Set the configured value intended for use in constructing new schedule
   * blocks.
   * @param {*} effectiveValue
   */
  DayEditor.prototype.$setDefaultEffectiveValue = function (effectiveValue) {
    this.properties().setValue('effectiveValue', effectiveValue);
  };

  /**
   * @returns {Array.<module:nmodule/schedule/rc/fe/ScheduleBlockEditor>}
   */
  DayEditor.prototype.$getScheduleBlocks = function () {
    return this.getChildWidgets(this.$getBlocksDisplayElement());
  };

  /**
   * Get the currently selected schedule block, if available.
   * @returns {module:nmodule/schedule/rc/model/ScheduleBlock|undefined}
   */
  DayEditor.prototype.$getSelectedBlock = function () {
    var selectedBlockEditor = this.$getSelectedBlockEditor();
    return selectedBlockEditor && selectedBlockEditor.value();
  };

  /**
   * Get the currently selected schedule block editor, if available.
   * @returns {module:nmodule/schedule/rc/fe/ScheduleBlockEditor|undefined}
   */
  DayEditor.prototype.$getSelectedBlockEditor = function () {
    return this.$getBlocksDisplayElement().find('.ScheduleBlockEditor.selected').data('widget');
  };

  /**
   * Set the currently selected block editor.
   * @param {module:nmodule/schedule/rc/fe/ScheduleBlockEditor} selectedBlock
   */
  DayEditor.prototype.$setSelectedBlock = function (selectedBlock) {
    this.$getScheduleBlocks().forEach(function (block) {
      block.jq().toggleClass('selected', block === selectedBlock);
    });
  };
  DayEditor.prototype.$getTranslator = function () {
    return this.properties().getValue('translator');
  };
  function reject(err) {
    return Promise.reject(new Error(err));
  }
  function newTime(millis) {
    return new TimeOfDay({
      millis: millis
    });
  }
  return DayEditor;
});
