/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
 * @module nmodule/schedule/rc/baja/fe/WeeklySchedulePropertiesBar
 */
define(['bajaux/util/CommandButton', 'bajaux/events', 'bajaux/mixin/responsiveMixIn', 'Promise', 'nmodule/js/rc/log/Log', 'nmodule/js/rc/switchboard/switchboard', 'nmodule/schedule/rc/baja/fe/ActiveDatesCalendar', 'nmodule/schedule/rc/baja/fe/DateScheduleEditor', 'nmodule/schedule/rc/fe/templates', 'nmodule/webEditors/rc/fe/fe', 'nmodule/webEditors/rc/fe/CompositeEditor', 'nmodule/webEditors/rc/fe/config/CompositeBuilder', 'nmodule/webEditors/rc/wb/commands/PopOutCommand'], function (CommandButton, events, responsiveMixIn, Promise, Log, switchboard, ActiveDatesCalendar, DateScheduleEditor, templates, fe, CompositeEditor, CompositeBuilder, PopOutCommand) {
  'use strict';

  var MODIFY_EVENT = events.MODIFY_EVENT,
    lexLabel = templates.lexLabel;
  var editorWithLexLabel = function editorWithLexLabel(key, lexKey) {
    return "\n    ".concat(lexLabel(lexKey || key), "\n    <div class=\"js-").concat(key, "\"></div>");
  };
  var tpl = function tpl(ed) {
    return "\n    <div class=\"row\">\n      <div class=\"column\">\n        <div class=\"effectiveStart\">\n          ".concat(editorWithLexLabel('effectiveStart'), "\n        </div>\n        <div class=\"effectiveFinish\">\n          ").concat(editorWithLexLabel('effectiveFinish'), "\n        </div>\n      </div>\n      <div class=\"column\">\n        <div class=\"cleanupExpiredEvents\">\n          ").concat(editorWithLexLabel('cleanupExpiredEvents', 'scheduler.cleanupSpecialEvents'), "\n        </div>\n        <div class=\"defaultOutput\">\n          ").concat(editorWithLexLabel('defaultOutput', 'scheduler.defaultOutput'), "\n        </div>\n        <div class=\"facets\">\n          ").concat(editorWithLexLabel('facets', 'scheduler.facets'), "\n          <button type=\"button\" class=\"popout ux-btn-tb\"></button>\n        </div>\n      </div>\n    </div>\n    <div class=\"row\">\n      <div class=\"activeDates\">\n        <div class=\"js-activeDates\"></div>\n      </div>\n    </div>");
  };

  /**
   * Toolbar to edit properties of a `WeeklySchedule`: `cleanupExpiredEvents`,
   * `defaultOutput`, `effective`, `facets`.
   *
   * @class
   * @alias module:nmodule/schedule/rc/fe/WeeklySchedulePropertiesBar
   * @extends module:nmodule/webEditors/rc/fe/CompositeEditor
   */
  var WeeklySchedulePropertiesBar = function WeeklySchedulePropertiesBar() {
    CompositeEditor.apply(this, arguments);
    switchboard(this, {
      $applyFacets: {
        notWhile: 'save',
        onRepeat: 'queue'
      },
      save: {
        notWhile: '$applyFacets',
        onRepeat: 'queue'
      }
    });
    responsiveMixIn(this, {
      /* Collapse properties into two groups arranged in two rows with calendar on third row */
      'weekly-schedule-properties-md': {
        maxWidth: 1067
      },
      /* Rearrange first row of properties to favor a vertical layout */
      'weekly-schedule-properties-sm': {
        maxWidth: 606
      },
      /* Rearrange second row of properties to favor a vertical layout */
      'weekly-schedule-properties-xs': {
        maxWidth: 430
      },
      /* Hide the prev / next page since at this point only one calendar displays */
      'weekly-schedule-properties-xxs': {
        maxWidth: 413
      }
    });
  };
  WeeklySchedulePropertiesBar.prototype = Object.create(CompositeEditor.prototype);
  WeeklySchedulePropertiesBar.prototype.constructor = WeeklySchedulePropertiesBar;

  /**
   * Element for the facets editor popout command.
   *
   * @private
   * @returns {jQuery}
   */
  WeeklySchedulePropertiesBar.prototype.$getFacetsPopOutElement = function () {
    return this.jq().find('.facets > .popout');
  };

  /**
   * Get the popout command button for the facets popout command.
   *
   * @private
   * @returns {module:bajaux/util/CommandButton}
   */
  WeeklySchedulePropertiesBar.prototype.$getFacetsPopOutCommandButton = function () {
    return this.$getFacetsPopOutElement().data('widget');
  };

  /**
   * Place editors for each property in predetermined elements in the
   * properties bar.
   *
   * @returns {module:nmodule/webEditors/rc/fe/config/CompositeBuilder}
   */
  WeeklySchedulePropertiesBar.prototype.makeBuilder = function () {
    var _this = this;
    var builder = new CompositeBuilder({
      removeOnDestroy: false
    });
    builder.getKeys = function () {
      return ['cleanupExpiredEvents', 'defaultOutput', 'facets', 'effectiveStart', 'effectiveFinish', 'activeDates'];
    };
    builder.getConfigFor = function (key) {
      var schedule = builder.getDataSource();
      switch (key) {
        case 'cleanupExpiredEvents':
          return {
            complex: schedule,
            slot: key,
            formFactor: 'mini'
          };
        case 'defaultOutput':
        case 'facets':
          return {
            complex: schedule,
            slot: key,
            properties: _this.$getFacets().toObject(),
            formFactor: 'mini'
          };
        case 'effectiveStart':
          return {
            complex: schedule.get('effective'),
            slot: 'start',
            type: DateScheduleEditor,
            formFactor: 'mini'
          };
        case 'effectiveFinish':
          return {
            complex: schedule.get('effective'),
            slot: 'end',
            type: DateScheduleEditor,
            formFactor: 'mini'
          };
        case 'activeDates':
          return {
            readonly: true,
            type: ActiveDatesCalendar,
            properties: {
              sourceOrd: _this.properties().getValue("sourceOrd")
            }
          };
      }
    };
    builder.getDomFor = function (key) {
      return _this.jq().find('.js-' + key);
    };
    builder.getValueFor = function (key) {
      var schedule = builder.getDataSource();
      switch (key) {
        case 'cleanupExpiredEvents':
        case 'defaultOutput':
        case 'facets':
          return schedule.get(key);
        case 'effectiveStart':
          return schedule.get('effective').get('start');
        case 'effectiveFinish':
          return schedule.get('effective').get('end');
        case 'activeDates':
          return schedule.get('effective');
      }
    };
    return builder;
  };

  /**
   * Add `WeeklySchedulePropertiesBar` class and set up DOM elements to hold
   * individual fields.
   *
   * @param {jQuery} dom
   */
  WeeklySchedulePropertiesBar.prototype.doInitialize = function (dom) {
    var _this2 = this;
    dom.addClass('WeeklySchedulePropertiesBar');
    dom.html(tpl(this));
    dom.on(MODIFY_EVENT, '.defaultOutput', function () {
      _this2.$readDefaultOutput().then(function (newValue) {
        return _this2.emit('defaultOutputChanged', newValue);
      })["catch"](Log.error);
    });
    dom.on(MODIFY_EVENT, '.facets', function () {
      _this2.getBuilder().getEditorFor('facets').read().then(function (facets) {
        return _this2.$applyFacets(facets);
      })["catch"](Log.error);
    });
    dom.on(MODIFY_EVENT, '.effectiveStart, .effectiveFinish', function () {
      _this2.$applyEffectiveRange()["catch"](Log.error);
    });
  };
  WeeklySchedulePropertiesBar.prototype.$readDefaultOutput = function () {
    var ed = this.getBuilder().getEditorFor('defaultOutput');
    return ed.read().then(function (diff) {
      return diff.apply(ed.value().newCopy());
    });
  };

  /**
   * Read the values for effective start/finish and apply them to the active
   * dates calendar.
   * @returns {Promise}
   */
  WeeklySchedulePropertiesBar.prototype.$applyEffectiveRange = function () {
    var builder = this.getBuilder(),
      startEd = builder.getEditorFor('effectiveStart'),
      finishEd = builder.getEditorFor('effectiveFinish'),
      activeDatesEd = builder.getEditorFor('activeDates'),
      effective = this.value().get('effective');
    return Promise.all([startEd.save(), finishEd.save()]).then(function () {
      return activeDatesEd.load(effective);
    });
  };
  WeeklySchedulePropertiesBar.prototype.$applyFacets = function (facets) {
    var _this3 = this;
    var builder = this.getBuilder(),
      defaultOutputKey = 'defaultOutput';
    this.properties().setValue('facets', facets);
    return builder.getEditorFor(defaultOutputKey).save().then(function () {
      return builder.destroyFor(defaultOutputKey);
    }).then(function () {
      return _this3.isInitialized() && builder.$loadFor(defaultOutputKey);
    }).then(function () {
      return Promise.all(_this3.emit('facetsChanged', facets));
    });
  };
  WeeklySchedulePropertiesBar.prototype.$getFacets = function () {
    return this.properties().getValue('facets');
  };

  /**
   * Load editors for each field and a command button to pop out details for
   * the facets editor.
   */
  WeeklySchedulePropertiesBar.prototype.doLoad = function (sched) {
    var _this4 = this;
    this.properties().add('facets', sched.get('facets'));
    return CompositeEditor.prototype.doLoad.apply(this, arguments).then(function () {
      //TODO: NCCB-27641: can we have smarter inter-editor dependency management?
      //CompositeBuilder doesn't allow me to declare editor A must wait for
      //editor B to exist before editor A can be created
      return fe.buildFor({
        dom: _this4.$getFacetsPopOutElement(),
        type: CommandButton,
        value: new PopOutCommand(_this4.getBuilder().getEditorFor('facets'))
      });
    });
  };
  WeeklySchedulePropertiesBar.prototype.doReadonly = function (readonly) {
    var commandButton = this.$getFacetsPopOutCommandButton(),
      commandEnable = !readonly && this.isEnabled(),
      commandValue = commandButton ? commandButton.value() : null;
    return Promise.join(commandValue && commandValue.setEnabled(commandEnable), CompositeEditor.prototype.doReadonly.apply(this, arguments));
  };
  WeeklySchedulePropertiesBar.prototype.doEnabled = function (enabled) {
    var commandButton = this.$getFacetsPopOutCommandButton(),
      commandEnable = enabled && !this.isReadonly(),
      commandValue = commandButton ? commandButton.value() : null;
    return Promise.join(commandValue && commandValue.setEnabled(commandEnable), CompositeEditor.prototype.doEnabled.apply(this, arguments));
  };

  /**
   * Remove `WeeklySchedulePropertiesBar` class and destroy popout button.
   */
  WeeklySchedulePropertiesBar.prototype.doDestroy = function () {
    this.jq().removeClass('WeeklySchedulePropertiesBar');
    var commandButton = this.$getFacetsPopOutCommandButton();
    return Promise.all([CompositeEditor.prototype.doDestroy.apply(this, arguments), commandButton && commandButton.destroy()]);
  };
  return WeeklySchedulePropertiesBar;
});
