/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/* eslint-env browser */
/**
 * API Status: **Private**
 * @module nmodule/schedule/rc/baja/fe/BajaWeekEditor
 */
define(['jquery', 'baja!', 'bajaux/commands/Command', 'bajaux/mixin/responsiveMixIn', 'bajaux/util/CommandButton', 'underscore', 'nmodule/js/rc/log/Log', 'nmodule/schedule/rc/baja/fe/BajaDayEditor', 'nmodule/schedule/rc/baja/fe/TimeScheduleEditor', 'nmodule/schedule/rc/baja/fe/WeeklySchedulePropertiesBar', 'nmodule/schedule/rc/baja/model/ScheduleTranslator', 'nmodule/schedule/rc/baja/util/bindDayEditorToTimeSchedule', 'nmodule/schedule/rc/fe/WeekEditor', 'nmodule/webEditors/rc/fe/CompositeEditor', 'nmodule/webEditors/rc/fe/config/CompositeBuilder'], function ($, baja, Command, responsiveMixIn, CommandButton, _, Log, BajaDayEditor, TimeScheduleEditor, WeeklySchedulePropertiesBar, ScheduleTranslator, bindDayEditorToTimeSchedule, WeekEditor, CompositeEditor, CompositeBuilder) {
  'use strict';

  var tpl = function tpl(ed) {
    return "\n    <div class=\"ux-btn-tb-group\">\n      <span class=\"editEvent ux-btn-tb\"></span>\n      <span class=\"toggleProperties ux-btn-tb\"></span>\n    </div>\n    <div class=\"properties collapsed\"></div>\n    <div class=\"week\"></div>\n    <div class=\"timeScheduleContainer\">\n      <div class=\"timeSchedule\"></div>\n    </div>";
  };
  var BajaWeekdaysEditor = function BajaWeekEditor() {
    WeekEditor.apply(this, arguments);
  };
  BajaWeekdaysEditor.prototype = Object.create(WeekEditor.prototype);
  BajaWeekdaysEditor.prototype.constructor = BajaWeekdaysEditor;

  /**
   * The builder just ensures that each day is a `BajaDayEditor`.
   * @returns {module:nmodule/webEditors/rc/fe/config/CompositeBuilder}
   */
  BajaWeekdaysEditor.prototype.makeBuilder = function () {
    var _arguments = arguments,
      _this = this;
    var builder = WeekEditor.prototype.makeBuilder.apply(this, arguments),
      getConfigFor = builder.getConfigFor;
    builder.getConfigFor = function () {
      var config = getConfigFor.apply(builder, _arguments);
      config.type = BajaDayEditor;
      config.properties = _.extend({}, config.properties, _this.properties().getValue('facets').toObject());
      return config;
    };
    return builder;
  };

  /**
   * Propagate all property changes down to child day editors.
   */
  BajaWeekdaysEditor.prototype.doChanged = function (name, value) {
    var builder = this.getBuilder(),
      keys = builder.getKeys();
    keys.forEach(function (k) {
      return builder.getEditorFor(k).properties().add(name, value);
    });
  };

  /**
   * This is a Niagara-specific editor to fill the WeeklySchedule tab when
   * viewing a Niagara schedule. It enforces Niagara-specific rules on each
   * weekday editor and provides a properties bar to edit Schedule properties.
   *
   * It should load in a {@link module:nmodule/schedule/rc/model/WeekSchedule},
   * and will read out an instance of the same.
   *
   * This editor uses the following `bajaux` `Properties`:
   *
   * - `snapshot`: {baja.Component} **required** - unmounted
   *   `schedule:WeeklySchedule` snapshot currently being edited in the
   *   scheduler
   *
   * @class
   * @alias module:nmodule/schedule/rc/baja/fe/BajaWeekEditor
   * @extends module:nmodule/schedule/rc/fe/WeekEditor
   */
  var BajaWeekEditor = function BajaWeekEditor() {
    var _this2 = this;
    CompositeEditor.apply(this, arguments);
    this.$togglePropertiesCommand = new Command({
      icon: 'module://icons/x16/gear.png',
      func: function func() {
        return _this2.getBuilder().getDomFor('properties').toggleClass('collapsed');
      }
    });
    this.$editEventCommand = new Command({
      icon: 'module://icons/x16/edit.png',
      func: function func() {
        return _this2.$getSelectedDay().$getEditCommand().invoke();
      },
      enabled: false
    });
    var props = this.properties();
    props.add('facets', props.getValue('snapshot').get('facets'));
    responsiveMixIn(this, {
      /* Hide the TimeSchedule and show the edit button */
      'week-editor-sm-y': {
        maxHeight: 360
      }
    });
  };
  BajaWeekEditor.prototype = Object.create(CompositeEditor.prototype);
  BajaWeekEditor.prototype.constructor = BajaWeekEditor;
  BajaWeekEditor.prototype.getResponsiveMediaInfo = function () {
    var container = this.jq().closest('.bajaux-widget-container');
    if (container.length) {
      if (!this.initialHeight) {
        this.initialHeight = Math.floor(container.innerHeight() || 0);
      }
      var height;
      if ($(document.activeElement) && $(document.activeElement).closest('.timeScheduleContainer').length) {
        height = this.initialHeight;
      } else {
        height = Math.floor(container.innerHeight() || 0);
      }
      return {
        width: Math.floor(container.innerWidth() || 0),
        height: height
      };
    } else {
      var w = $(window);
      return {
        width: w.width(),
        height: w.height()
      };
    }
  };
  BajaWeekEditor.prototype.makeBuilder = function () {
    var _this3 = this;
    var builder = new CompositeBuilder({
      removeOnDestroy: false
    });
    builder.getKeys = function () {
      return ['week', 'properties', 'editEvent', 'toggleProperties', 'timeSchedule'];
    };
    builder.getDomFor = function (key) {
      return _this3.jq().find('.' + key);
    };
    builder.getConfigFor = function (key) {
      switch (key) {
        case 'week':
          return {
            type: BajaWeekdaysEditor,
            properties: _this3.properties().clone()
          };
        case 'properties':
          return {
            type: WeeklySchedulePropertiesBar,
            properties: {
              sourceOrd: _this3.properties().getValue("sourceOrd")
            }
          };
        case 'editEvent':
          return {
            type: CommandButton
          };
        case 'toggleProperties':
          return {
            type: CommandButton
          };
        case 'timeSchedule':
          return {
            type: TimeScheduleEditor,
            readonly: true,
            properties: _this3.$getFacets().toObject()
          };
      }
    };
    builder.getValueFor = function (key) {
      switch (key) {
        case 'week':
          return _this3.value();
        case 'properties':
          return _this3.properties().getValue('snapshot');
        case 'editEvent':
          return _this3.$editEventCommand;
        case 'toggleProperties':
          return _this3.$togglePropertiesCommand;
        case 'timeSchedule':
          return _this3.$getDefaultTimeSchedule();
      }
    };
    return builder;
  };
  BajaWeekEditor.prototype.doInitialize = function (dom) {
    dom.html(tpl(this));
    return CompositeEditor.prototype.doInitialize.apply(this, arguments);
  };
  BajaWeekEditor.prototype.doLoad = function (value) {
    var _this4 = this;
    return CompositeEditor.prototype.doLoad.apply(this, arguments).then(function () {
      var builder = _this4.getBuilder(),
        propsEd = builder.getEditorFor('properties');
      propsEd.on('defaultOutputChanged', function (value) {
        return _this4.emit('defaultOutputChanged', value);
      });
      propsEd.on('facetsChanged', function (value) {
        _this4.properties().add('facets', value);
        _this4.emit('facetsChanged', value);
        var builder = _this4.getBuilder();
        builder.destroyFor('timeSchedule')
        //TODO: i got destroyed in the middle of a facetsChanged event. should CompositeBuilder have a destroy function too?
        .then(function () {
          return _this4.isInitialized() && builder.$loadFor('timeSchedule');
        })["catch"](Log.error);
      });
      var getDayEditor = function getDayEditor() {
          return _this4.$getSelectedDay();
        },
        getTimeScheduleEditor = function getTimeScheduleEditor() {
          return _this4.getBuilder().getEditorFor('timeSchedule');
        },
        defaultTimeSchedule = _this4.$getDefaultTimeSchedule();
      bindDayEditorToTimeSchedule(getDayEditor, getTimeScheduleEditor, defaultTimeSchedule, _this4.jq(), _this4.$editEventCommand);
    });
  };
  BajaWeekEditor.prototype.doRead = function () {
    return this.getBuilder().getEditorFor('week').read();
  };
  BajaWeekEditor.prototype.doSave = function () {
    return this.getBuilder().getEditorFor('properties').save();
  };

  /**
   * Propagate changes to the `facets` property down to the weekday editors.
   */
  BajaWeekEditor.prototype.doChanged = function (name, value) {
    switch (name) {
      case 'facets':
        return this.getBuilder().getEditorFor('week').properties().add(name, value);
    }
  };
  BajaWeekEditor.prototype.$getApplyMFCommand = function () {
    return this.getBuilder().getEditorFor('week').$getApplyMFCommand();
  };
  BajaWeekEditor.prototype.$getClearWeekCommand = function () {
    return this.getBuilder().getEditorFor('week').$getClearWeekCommand();
  };
  BajaWeekEditor.prototype.$getFacets = function () {
    return this.properties().getValue('facets') || baja.Facets.DEFAULT;
  };
  BajaWeekEditor.prototype.$getSelectedDay = function () {
    return _.find(this.getBuilder().getEditorFor('week').$getDayEditors(), function (dayEd) {
      return dayEd.$getSelectedBlockEditor();
    });
  };
  BajaWeekEditor.prototype.$getTranslator = function () {
    return new ScheduleTranslator(this.$getFacets());
  };
  BajaWeekEditor.prototype.$getDefaultTimeSchedule = function () {
    var timeSchedule = baja.$('schedule:TimeSchedule', {
      effectiveValue: this.properties().getValue('snapshot').get('out').newCopy()
    });
    timeSchedule.setFacets({
      slot: 'effectiveValue',
      facets: this.$getFacets()
    });
    return timeSchedule;
  };
  return BajaWeekEditor;
});
