/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
 * @module nmodule/bacnetOws/rc/wb/mgr/commands/GetEnrollmentSummaryCommand
 */
define(['baja!',
        'lex!bacnetOws',
        'jquery',
        'underscore',
        'dialogs',
        'bajaux/commands/Command',
        'nmodule/webEditors/rc/fe/feDialogs',
        'nmodule/webEditors/rc/wb/mgr/mgrUtils',
        'baja!bacnet:PriorityFilter,' +
             'bacnet:BacnetRecipientProcess,' +
             'bacnetOws:GetEnrollmentSummaryJob,' +
             'bacnetOws:GetESummConfig'], function (
        baja,
        lexs,
        $,
        _,
        dialogs,
        Command,
        feDialogs,
        mgrUtils) {

  'use strict';

  var getMainTableSelectedSubjects = mgrUtils.getMainTableSelectedSubjects,
      bacnetOwsLex = lexs[0];

////////////////////////////////////////////////////////////////
// GetEnrollmentSummaryCommand
////////////////////////////////////////////////////////////////

  function dlgTitle () {
    return bacnetOwsLex.get('getESumm.title');
  }

  function showFailureDialog (cause) {
    dialogs.showOk({
      title: dlgTitle(),
      content: function (dlg) {
        dlg.content().text(bacnetOwsLex.getSafe('getESumm.fail', cause));
      }
    });
  }

  function getFirstSelectedDevice (mgr) {
    var selection = getMainTableSelectedSubjects(mgr),
        folderType = mgr.getModel().getFolderType();

    function isNotFolder (c) {
      return !(c.getType().is(folderType));
    }

    return _.first(_(selection).filter(isNotFolder));
  }

  /**
   * Tokenize a single enrollment summary string and append it to the results string to
   * be displayed in a dialog.
   *
   * @param {string} results - the string being built containing all the combined enrollment summary strings.
   * @param {string} summary - an encoded enrollment summary response, tokenized by '|' characters.
   * @param {number} index - the index of the summary in the results array.
   * @returns {string} - the results string with the additional summary info appended.
   */
  function formatEnrollmentSummary (results, summary, index) {
    var tokens = summary.split('|');
    if (tokens.length < 5) { return 'Cannot parse: ' + summary; }

    if (index > 0) { results += '\n'; }

    return results + bacnetOwsLex.get({ key: 'getESumm.enrollmentSummaryFormat', args: tokens });
  }

  /**
   * Command type to invoke a BGetEnrollmentSummaryJob on the station and obtain the results
   * from dynamic slots on the job when it completes. Before submitting the job, this command
   * will display a dialog to get the user's choices for the 'getEnrollmentSummary' request.
   * This allows the user to set a filter for things such as a maximum/minimum priority
   * or particular notification class, for example.
   *
   * @class
   * @alias module:nmodule/bacnetOws/rc/wb/mgr/commands/GetEnrollmentSummaryCommand
   * @extends module:bajaux/commands/Command
   */
  var GetEnrollmentSummaryCommand = function GetEnrollmentSummaryCommand (mgr) {
    var that = this;

    Command.call(this, {
      displayName: bacnetOwsLex.get('getESumm.label'),
      icon: bacnetOwsLex.get('getESumm.icon'),
      description: bacnetOwsLex.get('getESumm.description'),
      enabled: false,

      /**
       * Display a dialog to let the user choose some filter criteria. Once this is done,
       * start a new job via an action on the network.
       *
       * @alias module:nmodule/bacnet/rc/wb/mgr/commands/GetEnrollmentSummaryCommand#invoke
       * @returns {Promise}
       */
      func: function () {
        var config = baja.$('bacnetOws:GetESummConfig');

        return feDialogs.showFor({
          value: config,
          properties: {
            showHeader: false,
            showFooter: false
          }
        })
        .then(function (diff) {
          var network = mgr.getNetwork(),
              device;

          if (diff) {
            device = getFirstSelectedDevice(mgr);

            if (device) {
              config.setDeviceAddress(device.getAddress().newCopy());

              return network.submitDeviceManagerJob(config)
                .then(function (ord) {
                  return mgr.setJob(baja.Ord.make({
                    base: baja.Ord.make('station:'),
                    child: ord.relativizeToSession()
                  }));
                });
            }
          }
        })
        .catch(function (err) {
          showFailureDialog(String(err));
          throw err;
        });
      }
    });

    mgr.on('jobcomplete', function (job) {
      if (job.getType().is('bacnetOws:GetEnrollmentSummaryJob')) {
        that.complete(job);
      }
    });
  };

  GetEnrollmentSummaryCommand.prototype = Object.create(Command.prototype);
  GetEnrollmentSummaryCommand.prototype.constructor = GetEnrollmentSummaryCommand;

  /**
   * Called when the submitted job is complete. Will obtain the results from
   * the encoded string properties of the job and display a dialog.
   *
   * @static
   * @param {baja.Component} job
   */
  GetEnrollmentSummaryCommand.prototype.complete = function (job) {
    return job.loadSlots()
      .then(function () {
        if ('success' === job.getJobState().getTag()) {
          dialogs.showOk({
            title: dlgTitle(),
            content: function (dlg) {
              var jq = dlg.content(),
                  summaries,
                  results;

              summaries = job.getSlots()
                             .properties()
                             .is('baja:String')
                             .toValueArray();

              results = bacnetOwsLex.get('getESumm.filter') +
                        formatFilterChoices(job.get('filter')) + '\n\n' +
                        _(summaries).reduce(formatEnrollmentSummary, '');

              jq.append($('<span>').text(bacnetOwsLex.get('getESumm.header') + '\n'))
                .append($('<pre>').text(results));
            }
          });
        }
        else {
          var cause = job.get('failureCause');
          if (cause === null) { cause = ''; }

          showFailureDialog(cause);
        }
      });
  };

////////////////////////////////////////////////////////////////
// Result Formatting
////////////////////////////////////////////////////////////////

  // The following are used when generating the results text for the dialog.
  // The Java version uses the toString() method on the types - rather than
  // create a number of type exts just for this dialog, the toString() methods
  // are just replicated here locally for now. These strings are also lexiconized,
  // which the Java versions are not.

  function isEnrollmentUsed (config) {
    var def = baja.$('bacnet:BacnetRecipientProcess');
    return !(config.getEnrollment().equivalent(def));
  }

  function formatEnrollmentFilter (enrollment) {
    return '\n  ' + bacnetOwsLex.get('getESumm.enrollmentFilter', enrollment.getRecipient(), enrollment.getProcessIdentifier());
  }

  function isEventStateUsed (config) {
    return config.getEventState().getTag() !== 'all';
  }

  function formatEventState (state) {
    return '\n  ' + bacnetOwsLex.get('getESumm.eventStateFilter', state.getTag());
  }

  function isEventTypeUsed (config) {
    return config.getEventType().getTag() !== 'all';
  }

  function formatEventTypeFilter (type) {
    return '\n  ' + bacnetOwsLex.get('getESumm.eventTypeFilter', type.getTag());
  }

  function isPriorityUsed (config) {
    var def = baja.$('bacnet:PriorityFilter');
    return !(config.getPriority().equivalent(def));
  }

  function formatPriorityFilter (priority) {
    return '\n  ' + bacnetOwsLex.get('getESumm.priorityFilter', priority.getMinPriority(), priority.getMaxPriority());
  }

  function isNotificationClassUsed (config) {
    return config.getNotificationClass() !== -1;
  }

  function formatNotificationClassFilter (nc) {
    return '\n  ' + bacnetOwsLex.get('getESumm.notificationClassFilter', nc);
  }

  function formatFilterChoices (config) {
    var str = '\n  ' + bacnetOwsLex.get('getESumm.ackFilter', config.getAcknowledgment().getTag());

    if (isEnrollmentUsed(config)) { str += formatEnrollmentFilter(config.getEnrollment()); }
    if (isEventStateUsed(config)) { str += formatEventState(config.getEventState()); }
    if (isEventTypeUsed(config)) { str += formatEventTypeFilter(config.getEventType()); }
    if (isPriorityUsed(config)) { str += formatPriorityFilter(config.getPriority()); }
    if (isNotificationClassUsed(config)) { str += formatNotificationClassFilter(config.getNotificationClass()); }

    return str;
  }

  return GetEnrollmentSummaryCommand;
});
