/**
 * @copyright 2019 Tridium, Inc. All Rights Reserved.
 * @author Andy Sutton
 */

define(['baja!', 'bajaScript/baja/tag/Id', 'nmodule/tagdictionary/rc/taginfo/TagInfoMixin', 'nmodule/tagdictionary/rc/taginfo/TagGroupInfoMixin'], function (baja, Id, TagInfoMixin, TagGroupInfoMixin) {
  'use strict';

  /**
   * A utility class for making and dealing with tagging related rpc calls
   *
   * API Status: **Private**
   * @exports nmodule/tagdictionary/rc/util/rpcUtil
   */
  var exports = {};
  var TAG_NAME_KEY = 'n',
    DICTIONARY_KEY = 'd',
    TAG_TYPE_KEY = 't',
    TAG_B_TYPE_KEY = 'b',
    DEFAULT_VALUE_KEY = 'v',
    IS_IDEAL_KEY = 'ii',
    IS_VALID_KEY = 'iv';
  var taggingRpc = function taggingRpc(methodName, argList, batch) {
    return baja.rpc({
      typeSpec: 'tagdictionary:TaggingRpcUtil',
      methodName: methodName,
      args: argList,
      batch: batch
    });
  };

  /**
   * Returns the ord for the Tag Dictionary service, returns an empty string if there isn't one.
   * @returns {Promise.<String>}
   */
  exports.getTagDictionaryServiceOrd = function () {
    return taggingRpc('getTagDictionaryServiceOrd', []);
  };

  /**
   * Retrieve the tagInfos for the given tagDictionary.
   * @param {String} tagDictionary
   * @param {String} targetComponentOrdStr
   * @returns {Promise.<Object>}
   */
  exports.getTagInfos = function (tagDictionary, targetComponentOrdStr) {
    return taggingRpc('getTagInfos', [tagDictionary, targetComponentOrdStr]).then(function (result) {
      var tagGroupInfos = [];
      result.forEach(function (tagInfoJson) {
        tagGroupInfos.push(convertToTagInfoMixin(tagInfoJson));
      });
      return tagGroupInfos;
    });
  };

  /**
   * Retrieve the tagGroupInfos for the given tagDictionary.
   * @param {String} tagDictionary
   * @returns {Promise.<Object>}
   */
  exports.getTagGroupInfos = function (tagDictionary, targetComponentOrdStr) {
    return taggingRpc('getTagGroupInfos', [tagDictionary, targetComponentOrdStr]).then(function (result) {
      var tagGroupInfos = [];
      result.forEach(function (tagGroupInfoJson) {
        tagGroupInfos.push(convertToTagGroupInfoMixin(tagGroupInfoJson));
      });
      return tagGroupInfos;
    });
  };
  function convertToTagInfoMixin(tagInfoJson) {
    return new TagInfoMixin(new Id(tagInfoJson[DICTIONARY_KEY], tagInfoJson[TAG_NAME_KEY]), baja.bson.decodeValue(JSON.parse(tagInfoJson[DEFAULT_VALUE_KEY])), tagInfoJson[TAG_B_TYPE_KEY], tagInfoJson[IS_IDEAL_KEY], tagInfoJson[IS_VALID_KEY], tagInfoJson[TAG_TYPE_KEY]);
  }
  function convertToTagGroupInfoMixin(tagGroupInfoJson) {
    return new TagGroupInfoMixin(new Id(tagGroupInfoJson[DICTIONARY_KEY], tagGroupInfoJson[TAG_NAME_KEY]), baja.bson.decodeValue(JSON.parse(tagGroupInfoJson[DEFAULT_VALUE_KEY])), tagGroupInfoJson[TAG_B_TYPE_KEY], tagGroupInfoJson[IS_IDEAL_KEY], tagGroupInfoJson[IS_VALID_KEY]);
  }
  return exports;
});
