/**
 * @copyright 2020 Tridium, Inc. All Rights Reserved.
 * @author Andy Sutton
 */

/**
 * A mixin that provides the functionality for the SelectedComponents table
 *
 * API Status: **Private**
 * @module nmodule/tagdictionary/rc/util/SelectedComponentsTableSupport
 */
define(['baja!', 'log!nmodule.tagdictionary.rc.util.SelectedComponentsTableSupport', 'Promise', 'underscore', 'jquery', 'nmodule/webEditors/rc/wb/mgr/Manager', 'nmodule/tagdictionary/rc/util/taggingUtil', 'nmodule/tagdictionary/rc/commands/RemoveComponentCommand', 'nmodule/tagdictionary/rc/model/SelectedComponentsModel', 'nmodule/webEditors/rc/wb/table/Table', 'nmodule/webEditors/rc/wb/table/menu/DefaultTableContextMenu', 'bajaux/util/CommandButtonGroup', 'bajaux/dragdrop/dragDropUtils', 'nmodule/webEditors/rc/fe/fe', 'nmodule/webEditors/rc/wb/mixin/mixinUtils', 'bajaux/commands/CommandGroup', 'lex!webEditors'], function (baja, log, Promise, _, $, Manager, taggingUtil, RemoveComponentCommand, SelectedComponentsModel, Table, DefaultTableContextMenu, CommandButtonGroup, dragDropUtils, fe, mixinUtils, CommandGroup, lexs) {
  'use strict';

  var applyMixin = mixinUtils.applyMixin,
    logError = log.severe.bind(log),
    MIXIN_NAME = 'SELECTED_COMPONENTS_TABLE',
    webEditorsLex = lexs[0];
  var hasWritePermissions = function hasWritePermissions(component) {
    return component.getPermissions().hasAdminWrite();
  };
  var typeSupportsTagging = function typeSupportsTagging(component) {
    return !baja.hasType(component, 'ace:AceComponent') && !baja.hasType(component, 'baja:VirtualComponent');
  };

  /**
   * A mixin to provide Selected Component Table support to Tag Manager views.
   *
   * @alias module:nmodule/tagdictionary/rc/SelectedComponentsModel
   * @mixin
   * @extends module:nmodule/tagdictionary/rc/TagUxManager
   * @param {module:nmodule/tagdictionary/rc/TagUxManager} target
   * @param {Object} params
   *
   */
  var SelectedComponentsTableSupport = function SelectedComponentsTableSupport(target, params) {
    if (!(target instanceof Manager)) {
      throw new Error('target must be a Manager instance.');
    }
    if (!applyMixin(target, MIXIN_NAME, SelectedComponentsTableSupport.prototype)) {
      return this;
    }
    var superDoInitialize = target.doInitialize,
      superDoLoad = target.doLoad,
      superDoDestroy = target.doDestroy;

    /**
     * Extend the base Manager's doInitialize() function,
     * adding the jobcomplete handler.
     *
     * @override
     * @param {JQuery} dom
     * @param {Object} [params]
     * @returns {*|Promise}
     */
    target.doInitialize = function (dom, params) {
      var that = this,
        selectedComponentRowsSelector = '.mgr-components-pane .ux-table-row',
        selectedComponentsPaneSelector = '.split-pane-component.mgr-pane-left';
      return superDoInitialize.call(that, dom, params || {}).then(function () {
        $('div.split-pane.vertical-percent').splitPane('firstComponentSize', dom.outerWidth() * 0.25);

        // set up drag and drop functionality for adding tags from the 'Available Tags' (discovery)
        // pane to the selected components in the 'Selected Components' pane.

        that.jq().on('dragover', selectedComponentRowsSelector, function (e) {
          // this has the effect of allowing drag and drop on a readonly input
          e.preventDefault();
        });
        that.jq().on('drop', selectedComponentRowsSelector, function (e) {
          that.$processDropFromDiscoveredTags(e);
        });

        // set up drag and drop functionality for adding components from the
        // nav tree to the 'Selected Components' pane.

        that.jq().on('dragover', selectedComponentsPaneSelector, function (e) {
          // this has the effect of allowing drag and drop on a readonly input
          e.preventDefault();
        });
        that.jq().on('drop', selectedComponentsPaneSelector, function (e) {
          that.$processDropFromNavTree(e)["catch"](logError);
        });

        // add a separate command group for any SelectedComponents table commands
        return fe.buildFor({
          dom: $('<div class="selected-comps-cmd-group"/>').prependTo(dom.find('.commandContainer.mgr-action-bar')),
          value: new CommandGroup({
            commands: [new RemoveComponentCommand(that)]
          }),
          type: CommandButtonGroup,
          properties: {
            toolbar: false
          }
        });
      }).then(function (ed) {
        that.$selectedComponentsCommandGroup = ed.value();
      });
    };

    /**
     * Extend the base Manager's load() function, looking up/populating SelectedComponentsTableSupport properties.
     *
     * @override
     * @param {baja.Component} component
     * @returns {Promise}
     */
    target.doLoad = function (component) {
      var that = this,
        args = arguments;
      return that.$loadSelectedComponentsTable().then(function () {
        return superDoLoad.apply(that, args);
      }).then(function () {
        // As a starting point, add the component that we're sitting on to the Selected Components model
        if (hasWritePermissions(component)) {
          return that.getSelectedComponentsModel().insertRows([that.getSelectedComponentsModel().makeComponentRow(component)]);
        }
      });
    };

    /**
     * Override #getTagManagerHtml to add the SelectedComponents pane to the Tag Manager view
     */
    target.getTagManagerHtml = function () {
      return taggingUtil.htmlForTaggingView();
    };

    /**
     * Update the count of objects in the title portion of the main database table pane.
     */
    target.updateSelectedComponentsDbObjectCount = function () {
      if (!this.isDestroyed()) {
        SelectedComponentsTableSupport.$updateRowCount(this.getSelectedComponentsModel(), this.jq().find('.components-table-object-count'));
      }
    };

    /**
     * Extend the manager's destroy functionality to clean up event handlers
     * relating to the selectedComponents table and model.
     *
     * @returns {*|Promise}
     */
    target.doDestroy = function () {
      this.jq().find('.split-pane.vertical-percent').off();
      return superDoDestroy.apply(this, arguments);
    };
  };

  /**
   * Update the object count text on one of the manager panes.
   *
   * @private
   * @param model the model requiring the count to be updated; either database or discovery.
   * @param jq the jquery object for the element to contain the row count.
   */
  SelectedComponentsTableSupport.$updateRowCount = function (model, jq) {
    // does the same as MgrLearnTableSupport#updateRowCount, but for the SelectedComponents pane
    jq.text(webEditorsLex.get({
      key: 'mgr.titlePane.objects',
      args: [model.getRows().length]
    }));
  };

  /**
   * Adds the tags selected from the AvailableTags pane to the components selected in
   * the SelectedComponents pane
   *
   * @private
   * @param {Event} event
   */
  SelectedComponentsTableSupport.prototype.$processDropFromDiscoveredTags = function (event) {
    var dataTransfer = event.originalEvent.dataTransfer,
      json = dataTransfer.getData('Text');
    var dropped, data;
    if (json && (dropped = JSON.parse(json)) && dropped.mime === 'niagara/strings') {
      data = dropped.data && JSON.parse(dropped.data);
      if (data && data.kind === 'mgr-drag-op' && data.source === 'mgr-learn') {
        this.$getAddCommand().invoke()["catch"](logError);
      }
    }
    event.preventDefault();
  };

  /**
   * Adds the component selected from the Nav Tree into the SelectedComponents pane
   *
   * @private
   * @param {Event} event
   * @returns  {Promise}
   */
  SelectedComponentsTableSupport.prototype.$processDropFromNavTree = function (event) {
    var that = this,
      dataTransfer = event.originalEvent.dataTransfer,
      json = dataTransfer.getData('Text');
    var dropped;
    if (json && (dropped = JSON.parse(json)) && dropped.mime !== 'niagara/navnodes') {
      // ignore if the mime type is not 'niagara/navnodes'
      return Promise.resolve();
    }
    return dragDropUtils.fromClipboard(event.originalEvent.dataTransfer).then(function (envelope) {
      return envelope.toValues();
    }).then(function (values) {
      var makeRowPromises = _.map(values, function (value) {
        return value.getNavOrd().get().then(function (comp) {
          if (typeSupportsTagging(comp) && hasWritePermissions(comp)) {
            return that.getSelectedComponentsModel().makeComponentRow(comp);
          }
        });
      });
      return Promise.all(makeRowPromises).then(function (rows) {
        return that.getSelectedComponentsModel().insertRows(_.compact(rows));
      });
    });
  };

  /**
   * gets the command for removing a component from the selectedComponents table
   *
   * @private
   * @returns {module:nmodule/tagdictionary/rc/commands/RemoveComponentCommand}
   */
  SelectedComponentsTableSupport.prototype.$getRemoveCommand = function () {
    // below is equivalent to mgrUtils#findCommand except that it operates
    // on $selectedComponentsCommandGroup instead of the standard manager CommandGroup
    return _.find(this.$selectedComponentsCommandGroup.getChildren(), function (child) {
      return child instanceof RemoveComponentCommand;
    });
  };

  /**
   * loads the selectedComponents table
   *
   * @private
   * @returns {Promise}
   */
  SelectedComponentsTableSupport.prototype.$loadSelectedComponentsTable = function () {
    var that = this;
    return SelectedComponentsModel.make(that).then(function (model) {
      return fe.buildFor({
        dom: $('<table class="ux-table no-stripe"></table>').appendTo(that.jq().find('.mgr-components-pane')),
        type: Table,
        value: model
      });
    }).then(function (ed) {
      that.$selectedComponentsTable = ed;
      var standHandleRowEvent = ed.$handleRowEvent;

      /**
       * Add to the standard functionality by setting the added row(s) to 'selected'
       *
       * @private
       * @param {module:nmodule/webEditors/rc/wb/table/model/TableModel} tableModel
       * @param {Array.<module:nmodule/webEditors/rc/wb/table/model/Row>} rows
       * @param {string} eventName
       * @param {Array.<*>} args
       * @returns {Promise}
       */
      that.$selectedComponentsTable.$handleRowEvent = function (tableModel, rows, eventName, args) {
        return standHandleRowEvent(tableModel, rows, eventName, args).then(function () {
          switch (eventName) {
            case 'rowsAdded':
              var startIndex = args[0];
              rows.forEach(function (row, rowIndex) {
                that.getSelectedComponentsSelection().toggle(startIndex + rowIndex);
              });
              break;
            case 'rowsRemoved':
              // set the first row as selected
              that.getSelectedComponentsSelection().toggle(0);
              break;
          }
          that.updateSelectedComponentsDbObjectCount();
        });
      };
      new DefaultTableContextMenu(ed).arm(that.jq(), '.mgr-show-hide-menu-components');
      that.updateSelectedComponentsDbObjectCount();
      that.getSelectedComponentsSelection().on('changed', _.bind(that.$selectedComponentsHandler, that));
    });
  };

  /**
   * update following a change in the SelectedComponents table.
   *
   * @private
   */
  SelectedComponentsTableSupport.prototype.$selectedComponentsHandler = function () {
    var selectedComponentsTableSelection = this.getSelectedComponentsSelection(),
      // selected & unselected rows
      firstSelectedRow = this.getSelectedComponentsTable().getSelectedRows()[0],
      selectedComponent = firstSelectedRow ? firstSelectedRow.getSubject() : undefined;

    // enable the RemoveCommand on the SelectedComponents pane if there are any components in it (selected or unselected)
    this.$getRemoveCommand().setEnabled(!selectedComponentsTableSelection.isEmpty());

    // update the 'Showing tags on' pane with the new (selected) component
    // (passes selectedComponent as undefined if there isn't a component to select)
    return this.updateShowTagsOnComponent(selectedComponent);
  };

  /**
   * return the SelectedComponents table
   * @returns {module:nmodule/webEditors/rc/wb/table/Table}
   */
  SelectedComponentsTableSupport.prototype.getSelectedComponentsTable = function () {
    return this.$selectedComponentsTable;
  };

  /**
   * return the SelectedComponents Model
   * @returns {module:nmodule/tagdictionary/rc/model/SelectedComponentsModel}
   */
  SelectedComponentsTableSupport.prototype.getSelectedComponentsModel = function () {
    return this.getSelectedComponentsTable() && this.getSelectedComponentsTable().value();
  };

  /**
   * return the ListSelection from the SelectedComponents table
   * @returns {module:nmodule/webEditors/rc/util/ListSelection}
   */
  SelectedComponentsTableSupport.prototype.getSelectedComponentsSelection = function () {
    return this.getSelectedComponentsTable().$getSelection();
  };
  return SelectedComponentsTableSupport;
});
