/**
 * @copyright 2020 Tridium, Inc. All Rights Reserved.
 * @author Andy Sutton
 */

/**
 * API Status: **Private**
 * @module nmodule/tagdictionary/rc/column/TagValueColumn
 */
define(['baja!', 'baja!baja:Marker,tagdictionary:TagGroupInfo', 'underscore', 'nmodule/webEditors/rc/wb/mgr/model/MgrColumn', 'nmodule/webEditors/rc/fe/baja/DisplayOnlyEditor', 'nmodule/webEditors/rc/fe/baja/util/compUtils'], function (baja, types, _, MgrColumn, DisplayOnlyEditor, compUtils) {
  'use strict';

  var MARKER_TYPE_NAME = baja.Marker.DEFAULT.getType().getTypeName();

  /**
   * @class
   * @alias module:nmodule/tagdictionary/rc/column/TagValueColumn
   * @extends module:nmodule/webEditors/rc/wb/mgr/model/MgrColumn
   * @param {Object} params
   */
  var TagValueColumn = function TagValueColumn(params) {
    MgrColumn.call(this, 'tagValue', params || {});
  };
  TagValueColumn.prototype = Object.create(MgrColumn.prototype);
  TagValueColumn.prototype.constructor = TagValueColumn;

  /**
   * Returns the value represented by this row.
   *
   * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row
   * @returns {*}
   */
  TagValueColumn.prototype.getValueFor = function (row) {
    var node = row.getTreeNode();
    if (node.isGroup()) {
      // a folder (not a Tag Group)
      return '';
    }
    if (node.isTagGroup()) {
      // Show 'Marker' as the value for an implied Tag Group.
      // (the tagGroupId is implied as a marker tag on the component)
      if (node.isImplied()) {
        return MARKER_TYPE_NAME;
      }

      // this should be a baja:Relation
      return row.getSubject().getSourceOrdDisplay();
    }

    // Use the subject's value if it has one (generally if this is a Tag.js object),
    // otherwise use the default value (tagdictionary:SimpleTagInfo or TagInfoMixin)
    var subject = row.getSubject(),
      tagValue = subject.getValue ? subject.getValue() : subject.getDefaultValue(),
      tagValueType = tagValue.getType();
    if (tagValueType.is('baja:Marker')) {
      return MARKER_TYPE_NAME;
    }
    return tagValue;
  };

  /**
   * Set the editor to readonly for a TagGroup, and for a Marker tag
   *
   * @param {Array.<module:nmodule/webEditors/rc/wb/table/model/Row>} rows the selected rows
   * @returns {Object}
   */
  TagValueColumn.prototype.getConfigFor = function (rows) {
    var config = MgrColumn.prototype.getConfigFor.apply(this, arguments);
    var subject = rows[0].getSubject();

    // make the value readonly is more than one row is selected, or for a TagGroup
    if (rows.length > 1 || subject.$isTagGroup || subject.getType().is('tagdictionary:TagGroupInfo')) {
      config = _.extend({
        readonly: true
      }, config);
    }

    // use the DisplayOnlyEditor for a baja.Marker
    if (subject.getDefaultValue && subject.getDefaultValue().getType().is('baja:Marker')) {
      config = _.extend({
        type: DisplayOnlyEditor
      }, config);
    }
    return config;
  };

  /**
   * We can't set multiple tags to the same tag value. So if editing more than
   * one row, we need the editor to be a DisplayOnlyEditor.
   * For one row, a DisplayOnlyEditor might be suitable (eg. for baja.Marker),
   * but it might not, so we return false and force the editor to re-found.
   *
   * @param {module:nmodule/webEditors/rc/fe/baja/BaseEditor} editor
   * @param {Array.<module:nmodule/webEditors/rc/wb/table/model/Row>} rows
   * @returns {boolean} true when: there's more than one row and it's a DisplayOnlyEditor
   */
  TagValueColumn.prototype.isEditorSuitable = function (editor, rows) {
    if (!editor || !editor.jq() || rows.length === 1) {
      return false;
    }

    // more than one row at this point ...
    return editor instanceof DisplayOnlyEditor;
  };

  /**
   * Override the returned proposed value for a baja.Marker
   *
   * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row
   * @returns {*}
   */
  TagValueColumn.prototype.getProposedValueFor = function (row) {
    if (row.getSubject().getDefaultValue && row.getSubject().getDefaultValue().getType().is('baja:Marker')) {
      return MARKER_TYPE_NAME;
    }
    return MgrColumn.prototype.getProposedValueFor.call(this, row);
  };

  /**
   * When the subject is a `baja.Complex`, set/add the property value on the row, then set the progress callback to `commitReady`.
   * When the subject is a not a `baja.Complex`, just set the progress callback to `commitReady` because there is nothing to commit.
   *
   * @param {baja.Value} value
   * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row
   * @param {Object} [params]
   * @param {baja.comm.Batch} [params.batch]
   * @returns {Promise}
   * @since Niagara 4.10U7
   */
  TagValueColumn.prototype.commit = function (value, row, params) {
    var subject = row.getSubject();
    var progressCallback = params && params.progressCallback,
      prom = Promise.resolve(subject instanceof baja.Complex && compUtils.writeSlot(subject, this.getName(), value, params));
    if (progressCallback) {
      progressCallback(MgrColumn.COMMIT_READY);
    }
    return prom;
  };
  return TagValueColumn;
});
