/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 */

/**
 * API Status: **Private**
 * @module nmodule/email/rc/mgr/EmailAccountManagerModel
 */
define(['baja!', 'underscore', 'Promise', 'nmodule/webEditors/rc/servlets/password', 'bajaScript/baja/obj/Password', 'nmodule/webEditors/rc/wb/mgr/model/MgrModel'], function (baja, _, Promise, password, Password, MgrModel) {
  'use strict';

  /**
   * Get component name.
   *
   * @inner
   * @param {baja.Component} comp
   * @returns {String}
   */
  function toName(comp) {
    return comp.getName();
  }

  /**
   * Get the email account password as a string, then reset the password to the
   * default so BajaScript will not attempt to encode it and send it over the
   * wire.
   *
   * @inner
   * @param {email:EmailAccount} account
   * @returns {String} Encoded password string
   */
  function extractAndResetPassword(account) {
    var pw = account.getPassword();
    account.setPassword(Password.DEFAULT);
    return pw.encodeToString();
  }

  /**
   * `MgrModel` for the Email Account Manager.
    * @class
   * @alias module:nmodule/email/rc/mgr/EmailAccountManagerModel
   * @extends module:nmodule/webEditors/rc/wb/mgr/model/MgrModel
   */
  var EmailAccountManagerModel = function EmailAccountManagerModel() {
    MgrModel.apply(this, arguments);
  };
  EmailAccountManagerModel.prototype = Object.create(MgrModel.prototype);
  EmailAccountManagerModel.prototype.constructor = EmailAccountManagerModel;

  /**
   * Before adding the email accounts to the station, wipe all the passwords off
   * (BajaScript will not encode them). After the accounts are added, send
   * the passwords up through the designated `PasswordRpcServlet`.
   *
   * @param {Array.<email:EmailAccount>} accounts
   * @returns {Promise}
   */
  EmailAccountManagerModel.prototype.addInstances = function (accounts) {
    var pwds = _.map(accounts, extractAndResetPassword),
      names = _.map(accounts, toName),
      emailService = this.getComponentSource().getContainer();
    return MgrModel.prototype.addInstances.apply(this, arguments).then(function () {
      return Promise.all(_(_.zip(names, pwds)).each(function (pair) {
        var name = pair[0],
          pwd = pair[1],
          account = emailService.get(name),
          slot = account.getSlot('password');
        return password.setPassword(pwd, slot, account);
      }));
    });
  };
  return EmailAccountManagerModel;
});
