/**
 * @copyright 2022 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
 * @module nmodule/httpClient/rc/PropertyPicker
 */
define([
  'baja!',
  'Promise',
  'underscore',
  'nmodule/webEditors/rc/fe/baja/BaseEditor'
], function (
  baja,
  Promise,
  _,
  BaseEditor
) {

  'use strict';

  /**
   * A field editor for picking from a list of
   * dynamic slots.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/httpClient/rc/PropertyPicker
   */
  var PropertyPicker = function PropertyPicker(params) {
    BaseEditor.call(this, _.extend({
      keyName: 'PropertyPicker'
    }, params));
  };
  PropertyPicker.prototype = Object.create(BaseEditor.prototype);
  PropertyPicker.prototype.constructor = PropertyPicker;

  /**
   * Initializes this editor.
   *
   * @param {JQuery} dom
   */
  PropertyPicker.prototype.doInitialize = function (dom) {
    var that = this;

    dom.addClass('PropertyPicker');

    dom.html(
      '<span><select class="property-selection" /></span>'
    );

    dom.on('change', '.property-selection', function (e) {
      that.setModified(true);
      e.preventDefault();
      return false;
    });
  };

  /**
   * Loads a list of property names, which are supplied as a csv list.
   *
   * @param {String} propertyList list of property names, which are supplied as a csv string.
   * @returns {Promise}
   */
  PropertyPicker.prototype.doLoad = function (propertyList) {
    this.$populateSelectOptions(propertyList);
  };

  /**
   * Reads the select element and return the currently selected value.
   *
   * @returns {String} the selected property name else an empty string.
   */
  PropertyPicker.prototype.doRead = function () {

    var selectedProperty = this.$getSelect().val();
    if (selectedProperty) {
      return selectedProperty;
    }
    return "";
  };

  /**
   * Enables or disables the select dropdown.
   *
   * @param {Boolean} enabled
   */
  PropertyPicker.prototype.doEnabled = function (enabled) {
    this.$getSelect().prop('disabled', this.isReadonly() || !enabled);
  };

  /**
   * Disables or enables the select dropdown.
   *
   * @param {Boolean} readonly
   */
  PropertyPicker.prototype.doReadonly = function (readonly) {
    this.$getSelect().prop('disabled', !this.isEnabled() || readonly);
  };

  /**
   * Clean up after this editor
   */
  PropertyPicker.prototype.doDestroy = function () {
    this.jq().removeClass('PropertyPicker');
  };

  /**
   * Populates the dropdown with the property names.
   *
   * @private
   * @param {String} propertyList list of property names, which are supplied as a csv string.
   */
  PropertyPicker.prototype.$populateSelectOptions = function (propertyList) {
    var that = this,
      selectElement = that.$getSelect();

    // clear out the dropdown options
    that.$getSelect().empty();

    var props = propertyList.split(',');

    // add a blank element to use if no values,
    // this element will be removed it the current value is an option.
    selectElement.append(
      '<option class="ux-option blank-option" value=""></option>'
    );
    // and make it the selected one.
    selectElement.val('');

    props.forEach(function (prop) {
      if (prop) { // TODO checks empty string?
        selectElement.append(
          '<option class="ux-option" value="' + _.escape(prop) + '">' + _.escape(baja.SlotPath.unescape(prop)) + '</option>');
      }
    });
  };

  /**
   * Get the dropdown select element.
   *
   * @private
   * @returns {JQuery}
   */
  PropertyPicker.prototype.$getSelect = function () {
    return this.jq().find('.property-selection');
  };

  return (PropertyPicker);
});
