/**
 * @copyright 2020 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
 * @module nmodule/httpClient/rc/HttpClientPointManager
 */
define([
  'baja!',
  'Promise',
  'underscore',
  'nmodule/ndriver/rc/NPointManager',
  'nmodule/driver/rc/wb/mgr/PointMgrModel',
  'nmodule/webEditors/rc/wb/mgr/MgrTypeInfo',
  'nmodule/httpClient/rc/PointPollCommand',
  'nmodule/httpClient/rc/AddMorePointsCommand',
  'nmodule/webEditors/rc/wb/mgr/mgrUtils',
  'css!nmodule/httpClient/rc/httpPointManager'
], function (
  baja,
  Promise,
  _,
  NPointManager,
  PointMgrModel,
  MgrTypeInfo,
  PointPollCommand,
  AddMorePointsCommand,
  mgrUtils
) {

  'use strict';

  var getMainTableSelection = mgrUtils.getMainTableSelection;

  /**
   * A bajaux Point Manager View for the HttpClientDriver
   *
   * @class
   * @extends module:nmodule/ndriver/rc/NPointManager
   * @alias module:nmodule/httpClient/rc/HttpClientPointManager
   *
   */
  var HttpClientPointManager = function HttpClientPointManager(params) {
    NPointManager.call(this, params);
  };

  HttpClientPointManager.prototype = Object.create(NPointManager.prototype);
  HttpClientPointManager.prototype.constructor = HttpClientPointManager;

  /**
   * Returns the set of Commands for the http client point manager.
   *
   * @returns {Array.<module:bajaux/commands/Command>}
   */
  HttpClientPointManager.prototype.makeCommands = function () {
    var that = this,
      retCmds = NPointManager.prototype.makeCommands.apply(that, arguments);

    retCmds.push(
      new PointPollCommand(that),
      new AddMorePointsCommand(that)
    );

    return retCmds;
  };

  /**
   * Update the selection state of the buttons on a given manager.
   *
   * @param manager the manager on which to check the button selection state.
   */
  HttpClientPointManager.prototype.updateButtonSelectionState = function (manager) {
    manager.getCommandGroup().getChildren().forEach(function (cmd) {
      if (cmd.$shouldBeEnabled) {
        cmd.setEnabled(cmd.$shouldBeEnabled(manager));
      }
    });
  };

  /**
   * Load the http client point manager view.
   *
   * {baja.Component} comp - the value being loaded into the manager.
   * @returns {Promise}
   */
  HttpClientPointManager.prototype.doLoad = function (comp) {
    var that = this;

    return NPointManager.prototype.doLoad.apply(that, arguments)
      .then(function () {

        var selection = getMainTableSelection(that);

        selection.on('changed', function () {
          that.updateButtonSelectionState(that);
        });
        that.updateButtonSelectionState(that);
      });
  };

  /**
   * Return `MgrTypeInfo` instances for the default new types for the Http Client point manager.
   *
   * @static
   * @returns {Promise.<Array.<module:nmodule/webEditors/rc/wb/mgr/MgrTypeInfo>>}
   */
  HttpClientPointManager.getDefaultNewTypes = function () {
    var types = [];

    PointMgrModel.addStringPointTypes(true, types);

    return MgrTypeInfo.make(types);
  };

  /**
   * Make the MgrModel for this Point Manager.
   *
   * @param {baja.Component} component - the NPointDeviceExt or NPointFolder that the Point Manager will display.
   * @returns {Promise.<module:nmodule/webEditors/rc/wb/mgr/model/MgrModel>}
   */
  HttpClientPointManager.prototype.makeModel = function (component) {
    return Promise.all([
      NPointManager.prototype.makeModel.apply(this, arguments),
      HttpClientPointManager.getDefaultNewTypes()
    ]).spread(function (model, overrideNewTypes) {
      model.$newTypes = overrideNewTypes;
      return model;
    });
  };

  return HttpClientPointManager;
});
