/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 */

/* jshint browser: true */

/**
 * API Status: **Private**
 * @module nmodule/bacnetAws/rc/wb/mgr/commands/RestoreCommand
 */
define(['baja!',
  'lex!bacnetAws',
  'Promise',
  'underscore',
  'dialogs',
  'nmodule/js/rc/csrf/csrfUtil',
  'nmodule/webEditors/rc/wb/mgr/mgrUtils',
  'bajaux/commands/Command',
  'nmodule/webEditors/rc/fe/feDialogs',
  'nmodule/bacnetAws/rc/wb/mgr/commands/BacnetCommandSheet',
  'nmodule/webEditors/rc/fe/baja/FileInputEditor',
  'baja!bacnetAws:RestoreConfig,bacnetAws:RestoreJob,bacnetAws:BacnetAwsNetwork'], function (baja,
                                                                                             lexs,
                                                                                             Promise,
                                                                                             _,
                                                                                             dialogs,
                                                                                             csrfUtil,
                                                                                             mgrUtils,
                                                                                             Command,
                                                                                             feDialogs,
                                                                                             BacnetCommandSheet,
                                                                                             FileInputEditor) {

  'use strict';

  var bacnetAwsLex      = lexs[0],
      getTableSelection = mgrUtils.getMainTableSelectedSubjects;

  /**
   * Return an object to be displayed in a property sheet for the Comm control config.
   * This will use a couple of dynamic enums, which will be converted to a instance
   * of BCommControlConfig later.
   *
   * @returns {Promise<baja.Component>}
   */
  function makeConfigObject() {
    var config = baja.$("bacnetAws:RestoreConfig"),
        comp   = toComponent(config);

    return Promise.resolve(comp);
  }

  /**
   * Transfer the Struct slot values to a Component slot values as dynamicProperties
   * @param {baja.Struct} config
   * @returns {baja.Component}
   */
  function toComponent(config) {
    var comp = baja.$("baja:Component");
    var nameMap = {};

    comp.add({
      slot: "restoreFile",
      value: "",
      facets: baja.Facets.make(['uxFieldEditor'], ['nmodule/webEditors/rc/fe/baja/FileInputEditor'])
    });

    nameMap.restoreFile = bacnetAwsLex.get('restoreFile');

    config.getSlots().properties().each(function (prop) {

      var propName    = prop.getName(),
          facets      = config.getFacets(propName),
          flags       = config.getFlags(propName),
          displayName = config.getDisplayName(propName),
          value       = config.get(propName).newCopy();

      if (prop.getName() === "password") {
        facets = baja.Facets.make(facets, baja.Facets.make(['uxFieldEditor'], ['nmodule/webEditors/rc/fe/baja/UnmountedPasswordEditor']));
      }

      if (prop.getName() === "directory") {
        flags = flags | baja.Flags.HIDDEN;
      }

      comp.add({
        slot: propName,
        value: value,
        facets: facets,
        flags: flags
      });
      nameMap[propName] = displayName;
    });

    comp.add({
      slot: "displayNames",
      flags: baja.Flags.HIDDEN | baja.Flags.READONLY,
      value: baja.NameMap.make(nameMap)
    });
    return comp;
  }

  /**
   * Transfer the component slot values to the struct slot values
   * @param {baja.Struct} struct
   * @param {baja.Component} comp
   */
  function toStruct(struct, comp) {
    struct.getSlots().properties().each(function (prop) {
      struct.set(
        {
          slot: prop.getName(),
          value: comp.get(prop.getName()).newCopy()
        });
    });
    return struct;
  }

  /**
   * Show the given configuration object in a property sheet. The
   * values will be used to set the argument for the job.
   *
   * @returns {Promise}
   */
  function showConfigDialog(config) {
    return Promise.resolve().then(function () {
      return feDialogs.showFor({
        title: bacnetAwsLex.get('restore.title'),
        type: BacnetCommandSheet,
        value: config,
        properties: {
          showHeader: false,
          showFooter: false
        }
      });
    });
  }

  /**
   * Display an error dialog, inserting the error text into the failure message.
   */
  function showErrorDialog(err) {
    baja.error(err);
    dialogs.showOk({
      title: bacnetAwsLex.get('restore.title'),
      content: bacnetAwsLex.getSafe({ key: 'restore.fail', args: [ err ] })
    });
  }

  function marshal(object) {
    var password = object.get('password').valueOf();
    object.set({
      slot: 'password',
      value: baja.$('baja:Password')
    }); //empty the password
    return [JSON.stringify(baja.bson.encodeValue(object)), password];
  }

  /**
   * Command for firing a ReinitializeDevice on the BACnet AWS network.
   *
   * @class
   * @alias module:nmodule/bacnetAws/rc/wb/mgr/commands/RestoreCommand
   * @extends module:bajaux/commands/Command
   */
  var RestoreCommand = function RestoreCommand(mgr) {
    var that = this;

    Command.call(this, {
      displayName: bacnetAwsLex.get('restore.label'),
      icon: bacnetAwsLex.get('restore.icon'),
      enabled: false,

      /**
       * Display a dialog with the choices of RestoreConfig parameters.
       *
       * @alias module:nmodule/bacnetAws/rc/wb/mgr/commands/RestoreCommand#invoke
       * @returns {Promise}
       */
      func: function () {
        var network = mgr.getNetwork();


        var original;
        return makeConfigObject()
          .then(function (c) {
            original = c;
            return showConfigDialog(c);
          })
          .then(function (diff) {

            if (diff) {
              return original;
            }
          })
          .then(function (config) {

            if (!config) {
              return;
            }
            var jobConfig = toStruct(baja.$('bacnetAws:RestoreConfig'), config);

            //add device specific info
            var selection = getTableSelection(mgr)[0];
            jobConfig.set({
              slot: 'deviceAddress',
              value: selection.getAddress().newCopy()
            });
            jobConfig.set({
              slot: 'characterSet',
              value: selection.getCharacterSet()
            });

            jobConfig.set({
              slot: 'deviceId',
              value: selection.getConfig().getDeviceObject().getObjectId()
            });

            var args     = marshal(jobConfig),
                formData = new FormData();

            formData.append("networkOrd", String(network.getNavOrd()));
            formData.append("config", args[0]);
            formData.append("password", args[1]);
            formData.append("file", FileInputEditor.getLastUploadedFile());
            //
            var xhr = new XMLHttpRequest();
            xhr.open("POST", "/bacnetAws/bacnetAws/restore");
            xhr.setRequestHeader("x-niagara-csrfToken", csrfUtil.getCsrfToken());
            xhr.onreadystatechange = function () {
              if (xhr.readyState === XMLHttpRequest.DONE) {
                var ord = baja.Ord.make(xhr.responseText);

                // Pass the job to the manager. This will set it on the job bar and
                // will listen for the completion event. It will check the type and
                // display a message when complete.

                mgr.setJob(baja.Ord.make({
                  base: baja.Ord.make('station:'),
                  child: ord.relativizeToSession()
                }));
              }
            };
            xhr.send(formData);

          })
          .catch(showErrorDialog);
      }
    });

    mgr.on('jobcomplete', function (job) {
      if (job.getType().is('bacnetAws:RestoreJob')) {
        that.complete(job);
      }
    });
  };

  RestoreCommand.prototype = Object.create(Command.prototype);
  RestoreCommand.prototype.constructor = RestoreCommand;

  /**
   * Function called when the RestoreCommand has completed.
   *
   * @param {baja.Component} job - the time synch job, submitted by the invoke function
   * above.
   */
  RestoreCommand.prototype.complete = function (job) {
    var msg, cause;

    if ('success' === job.getJobState().getTag()) {
      msg = bacnetAwsLex.getSafe('restore.success');
    }
    else {
      cause = job.get('failureCause') || '';
      msg = bacnetAwsLex.getSafe({key: 'restore.fail', args: [cause]});
    }

    dialogs.showOk({
      title: bacnetAwsLex.get('restore.title'),
      content: msg
    });
  };

  return RestoreCommand;
})
;
