/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
 * @module nmodule/bacnetAws/rc/wb/mgr/commands/DeviceCommunicationControlCommand
 */
define(['baja!',
  'lex!bacnetAws',
  'Promise',
  'dialogs',
  'nmodule/webEditors/rc/wb/mgr/mgrUtils',
  'bajaux/commands/Command',
  'nmodule/webEditors/rc/fe/feDialogs',
  'nmodule/bacnetAws/rc/wb/mgr/commands/BacnetCommandSheet',
  'baja!bacnetAws:CommControlConfig,bacnetAws:DeviceCommControlJob,bacnetAws:BacnetAwsNetwork'], function (baja,
                                                                                                           lexs,
                                                                                                           Promise,
                                                                                                           dialogs,
                                                                                                           mgrUtils,
                                                                                                           Command,
                                                                                                           feDialogs,
                                                                                                           BacnetCommandSheet) {

  'use strict';

  var bacnetAwsLex      = lexs[0],
      getTableSelection = mgrUtils.getMainTableSelectedSubjects;

  /**
   * Return an object to be displayed in a property sheet for the Comm control config.
   * This will use a couple of dynamic enums, which will be converted to a instance
   * of BCommControlConfig later.
   *
   * @returns {Promise<baja.Component>}
   */
  function makeConfigObject() {
    var config = baja.$("bacnetAws:CommControlConfig"),
        comp   = toComponent(config);

    return Promise.resolve(comp);
  }

  /**
   * Transfer the Struct slot values to a Component slot values as dynamicProperties
   * @param {baja.Struct} config
   * @returns {baja.Component}
   */
  function toComponent(config) {
    var comp    = baja.$("baja:Component"),
        nameMap = {};
    config.getSlots().properties().each(function (prop) {

      var name   = prop.getName(),
          facets = config.getFacets(name);
      if (name === "password") {
        facets = baja.Facets.make(facets, baja.Facets.make(['uxFieldEditor'], ['nmodule/webEditors/rc/fe/baja/UnmountedPasswordEditor']));
      }
      comp.add({
        slot: name,
        value: config.get(name).newCopy(),
        facets: facets,
        flags: config.getFlags(name)
      });
      nameMap[name] = config.getDisplayName(name);
    });

    comp.add({
      slot: "displayNames",
      flags: baja.Flags.HIDDEN | baja.Flags.READONLY,
      value: baja.NameMap.make(nameMap)
    });
    return comp;
  }

  /**
   * Transfer the component slot values to the struct slot values
   * @param {baja.Struct} struct
   * @param {baja.Component} comp
   */
  function toStruct(struct, comp) {
    struct.getSlots().properties().each(function (prop) {
      var name = prop.getName();
      struct.set(
        {
          slot: name,
          value: comp.get(name).newCopy()
        });
    });
    return struct;
  }

  /**
   * Show the given configuration object in a property sheet. The
   * values will be used to set the argument for the job.
   *
   * @returns {Promise}
   */
  function showConfigDialog(config) {
    return Promise.resolve().then(function () {
      return feDialogs.showFor({
        title: bacnetAwsLex.get('devCommCtrl.title'),
        type: BacnetCommandSheet,
        value: config,
        properties: {
          showHeader: false,
          showFooter: false
        }
      });
    });
  }

  /**
   * Display an error dialog, inserting the error text into the failure message.
   */
  function showErrorDialog(err) {
    baja.error(err);
    dialogs.showOk({
      title: bacnetAwsLex.get('devCommCtrl.title'),
      content: bacnetAwsLex.getSafe({ key: 'devCommCtrl.fail', args: [ err ] })
    });
  }

  function marshal(object) {
    var password = object.get('password').valueOf();
    object.set({
      slot: 'password',
      value: baja.$('baja:Password')
    }); //empty the password
    return [JSON.stringify(baja.bson.encodeValue(object)), password];
  }

  /**
   * Command for firing a DeviceCommunicationControl on the BACnet AWS network.
   *
   * @class
   * @alias module:nmodule/bacnetAws/rc/wb/mgr/commands/DeviceCommunicationControlCommand
   * @extends module:bajaux/commands/Command
   */
  var DeviceCommunicationControlCommand = function DeviceCommunicationControlCommand(mgr) {
    var that = this;

    Command.call(this, {
      displayName: bacnetAwsLex.get('devCommCtrl.label'),
      icon: bacnetAwsLex.get('devCommCtrl.icon'),
      enabled: false,

      /**
       * Display a dialog with the choices of CommControlConfig parameters.
       *
       * @alias module:nmodule/bacnetAws/rc/wb/mgr/commands/DeviceCommunicationControlCommand#invoke
       * @returns {Promise}
       */
      func: function () {
        var network = mgr.getNetwork();


        var original;
        return makeConfigObject()
          .then(function (c) {
            original = c;
            return showConfigDialog(c);
          })
          .then(function (diff) {

            if (diff) {
              return original;
            }
          })
          .then(function (config) {

            if (!config) {
              return;
            }
            var jobConfig = toStruct(baja.$('bacnetAws:CommControlConfig'), config);

            //add device specific info
            var selection = getTableSelection(mgr);
            jobConfig.set({
              slot: 'deviceAddress',
              value: selection[0].getAddress().newCopy()
            });
            jobConfig.set({
              slot: 'characterSet',
              value: selection[0].getCharacterSet()
            });

            return baja.rpc({
              ord: network.getNavOrd(),
              method: 'doSubmitDeviceManagerJob',
              args: marshal(jobConfig)
            })
              .then(function (ord) {

                //rpc returns a string, not an ord
                ord = baja.Ord.make(ord);

                // Pass the job to the manager. This will set it on the job bar and
                // will listen for the completion event. It will check the type and
                // display a message when complete.

                mgr.setJob(baja.Ord.make({
                  base: baja.Ord.make('station:'),
                  child: ord.relativizeToSession()
                }));
              });
          })
          .catch(showErrorDialog);
      }
    });

    mgr.on('jobcomplete', function (job) {
      if (job.getType().is('bacnetAws:DeviceCommControlJob')) {
        that.complete(job);
      }
    });
  };

  DeviceCommunicationControlCommand.prototype = Object.create(Command.prototype);
  DeviceCommunicationControlCommand.prototype.constructor = DeviceCommunicationControlCommand;

  /**
   * Function called when the DeviceCommunicationControlCommand has completed.
   *
   * @param {baja.Component} job - the time synch job, submitted by the invoke function
   * above.
   */
  DeviceCommunicationControlCommand.prototype.complete = function (job) {
    var msg, cause;

    if ('success' === job.getJobState().getTag()) {
      msg = bacnetAwsLex.getSafe('devCommCtrl.success');
    }
    else {
      cause = job.get('failureCause') || '';
      msg = bacnetAwsLex.getSafe({key: 'devCommCtrl.fail', args: [cause]});
    }

    dialogs.showOk({
      title: bacnetAwsLex.get('devCommCtrl.title'),
      content: msg
    });
  };

  return DeviceCommunicationControlCommand;
})
;
