/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 */

/*jshint browser: true */

/**
 * API Status: **Private**
 * @module nmodule/bacnetAws/rc/wb/mgr/commands/BackupCommand
 */
define(['baja!',
  'lex!bacnetAws',
  'jquery',
  'Promise',
  'dialogs',
  'moment',
  'nmodule/js/rc/csrf/csrfUtil',
  'nmodule/webEditors/rc/wb/mgr/mgrUtils',
  'bajaux/commands/Command',
  'nmodule/webEditors/rc/fe/feDialogs',
  'nmodule/bacnetAws/rc/wb/mgr/commands/BacnetCommandSheet',
  'baja!bacnetAws:BackupConfig,bacnetAws:BackupJob,bacnetAws:BacnetAwsNetwork'], function (baja,
                                                                                           lexs,
                                                                                           $,
                                                                                           Promise,
                                                                                           dialogs,
                                                                                           moment,
                                                                                           csrfUtil,
                                                                                           mgrUtils,
                                                                                           Command,
                                                                                           feDialogs,
                                                                                           BacnetCommandSheet) {

  'use strict';

  var bacnetAwsLex      = lexs[0],
      getTableSelection = mgrUtils.getMainTableSelectedSubjects;

  /**
   * Return an object to be displayed in a property sheet for the Comm control config.
   * This will use a couple of dynamic enums, which will be converted to a instance
   * of BCommControlConfig later.
   *
   * @returns {Promise<baja.Component>}
   */
  function makeConfigObject() {
    var config = baja.$("bacnetAws:BackupConfig"),
        comp   = toComponent(config);

    return Promise.resolve(comp);
  }

  /**
   * Transfer the Struct slot values to a Component slot values as dynamicProperties
   * @param {baja.Struct} config
   * @returns {baja.Component}
   */
  function toComponent(config) {
    var comp = baja.$("baja:Component");
    var nameMap = {};
    config.getSlots().properties().each(function (prop) {

      var propName    = prop.getName(),
          facets      = config.getFacets(propName),
          flags       = config.getFlags(propName),
          displayName = config.getDisplayName(propName),
          value       = config.get(propName).newCopy();

      if (prop.getName() === "password") {
        facets = baja.Facets.make(facets, baja.Facets.make(['uxFieldEditor'], ['nmodule/webEditors/rc/fe/baja/UnmountedPasswordEditor']));
      }

      if (prop.getName() === "baseDirectory") {
        flags = flags | baja.Flags.HIDDEN;
      }

      if (prop.getName() === "deviceDirectoryName") {
        displayName = bacnetAwsLex.get('downloadFileName');
      }

      comp.add({
        slot: propName,
        value: value,
        facets: facets,
        flags: flags
      });
      nameMap[propName] = displayName;
    });

    comp.add({
      slot: "displayNames",
      flags: baja.Flags.HIDDEN | baja.Flags.READONLY,
      value: baja.NameMap.make(nameMap)
    });
    return comp;
  }

  /**
   * Transfer the component slot values to the struct slot values
   * @param {baja.Struct} struct
   * @param {baja.Component} comp
   */
  function toStruct(struct, comp) {
    struct.getSlots().properties().each(function (prop) {
      struct.set(
        {
          slot: prop.getName(),
          value: comp.get(prop.getName()).newCopy()
        });
    });
    return struct;
  }

  /**
   * Show the given configuration object in a property sheet. The
   * values will be used to set the argument for the job.
   *
   * @returns {Promise}
   */
  function showConfigDialog(config) {
    return feDialogs.showFor({
      title: bacnetAwsLex.get('backup.title'),
      type: BacnetCommandSheet,
      value: config,
      properties: {
        showHeader: false,
        showFooter: false
      }
    });
  }

  /**
   * Display an error dialog, inserting the error text into the failure message.
   */
  function showErrorDialog(err) {
    baja.error(err);
    dialogs.showOk({
      title: bacnetAwsLex.get('backup.title'),
      content: bacnetAwsLex.getSafe({ key: 'backup.fail', args: [ err ] })
    });
  }

  function marshal(object) {
    var password = object.get('password').valueOf();
    object.set({
      slot: 'password',
      value: baja.$('baja:Password')
    }); //empty the password
    return [JSON.stringify(baja.bson.encodeValue(object)), password];
  }

  /**
   * Command for firing a Backup on the BACnet AWS network.
   *
   * @class
   * @alias module:nmodule/bacnetAws/rc/wb/mgr/commands/BackupCommand
   * @extends module:bajaux/commands/Command
   */
  var BackupCommand = function BackupCommand(mgr) {
    var that = this;

    Command.call(this, {
      displayName: bacnetAwsLex.get('backup.label'),
      icon: bacnetAwsLex.get('backup.icon'),
      enabled: false,

      /**
       * Display a dialog with the choices of BackupConfig parameters.
       *
       * @alias module:nmodule/bacnetAws/rc/wb/mgr/commands/BackupCommand#invoke
       * @returns {Promise}
       */
      func: function () {
        var network   = mgr.getNetwork(),
            selection = getTableSelection(mgr)[0];


        var original;
        return makeConfigObject()
          .then(function (c) {
            original = c;
            c.set({
              slot: 'deviceDirectoryName',
              value: selection.getName() + moment().format('YYYYMMDD_HHmm')
            });

            return showConfigDialog(c);
          })
          .then(function (diff) {

            //If OK was pressed, a Complex Diff will be given as the result, but the original object was already applied.
            if (diff) {
              return original;
            }
          })
          .then(function (config) {

            if (!config) {
              return;
            }
            var jobConfig = toStruct(baja.$('bacnetAws:BackupConfig'), config);

            //add device specific info

            jobConfig.set({
              slot: 'deviceAddress',
              value: selection.getAddress().newCopy()
            });
            jobConfig.set({
              slot: 'characterSet',
              value: selection.getCharacterSet()
            });

            jobConfig.set({
              slot: 'deviceId',
              value: selection.getConfig().getDeviceObject().getObjectId()
            });

            var args = marshal(jobConfig);

            return that.ajax("/bacnetAws/bacnetAws/backup", {
              type: "POST",
              data: {
                networkOrd: String(network.getNavOrd()),
                config: args[0],
                password: args[1]
              }
            }).then(function (response) {

              var split = response.split('\n');
              //rpc returns a string, not an ord
              var ord          = baja.Ord.make(baja.SlotPath.unescape(split[0])),
                  downloadName = baja.SlotPath.unescape(split[1]);


              // Pass the job to the manager. This will set it on the job bar and
              // will listen for the completion event. It will check the type and
              // display a message when complete.

              var jobOrd = baja.Ord.make({
                base: baja.Ord.make('station:'),
                child: ord.relativizeToSession()
              });
              mgr.setJob(jobOrd);


              //before providing a download, give some time to ensure job doesn't fail early in the job for things like password failures
              return jobOrd.get({lease: true}).then(function (job) {
                setTimeout(function () {
                  var tag = job.getJobState().getTag();
                  if (tag === 'running' || tag === 'success') {
                    //download the backup via the BacnetAwsServelet backup
                    //backup is considered a sensitive download, so require csrf token
                    var requestString = "/bacnetAws/bacnetAws/download/" + baja.SlotPath.escape(ord.toString());
                    requestString += "/" + baja.SlotPath.escape(downloadName + '.zip');
                    requestString += "?csrfToken=" + csrfUtil.getCsrfToken();
                    that.download(requestString);
                  }
                }, 1000);
              });
            });
          })
          .catch(showErrorDialog);
      }
    });

    mgr.on('jobcomplete', function (job) {
      if (job.getType().is('bacnetAws:BackupJob')) {
        that.complete(job);
      }
    });
  };

  BackupCommand.prototype = Object.create(Command.prototype);
  BackupCommand.prototype.constructor = BackupCommand;


  /**
   * Backup currently does not work in workbench due to
   * NCCB-7155: Download functionality is currently not implemented in JavaFx WebView
   */
  BackupCommand.prototype.setEnabled = function () {
    var isWb = window.niagara &&
      window.niagara.env &&
      window.niagara.env.type === "wb";

    if (!isWb) {
      Command.prototype.setEnabled.apply(this, arguments);
    }
  };

  /**
   * Change window location to new url for download
   * @param {String} newLocation for download
   */
  BackupCommand.prototype.download = function (newLocation) {
    window.location=newLocation;
  };


  /**
   * Function called when the BackupCommand has completed.
   *
   * @param {baja.Component} job - the time synch job, submitted by the invoke function
   * above.
   */
  BackupCommand.prototype.complete = function (job) {
    var msg, cause;

    if ('success' === job.getJobState().getTag()) {
      msg = bacnetAwsLex.getSafe('backup.success');
    }
    else {
      cause = job.get('failureCause') || '';
      msg = bacnetAwsLex.getSafe({key: 'backup.fail', args: [cause]});
    }

    dialogs.showOk({
      title: bacnetAwsLex.get('backup.title'),
      content: msg
    });
  };

  /**
   * Request an ajax call to a servlet and use proper error handling.
   * @param {String} uri
   * @param {Object} [params]
   * @returns {Promise}
   */
  BackupCommand.prototype.ajax = function (uri, params) {
    return new Promise(function (resolve, reject) {
      if (!params) {
        params = {};
      }

      if (!params.type) {
        params.type = "GET";
      }

      params.success = function (result) {
        resolve(result);
      };

      params.error = function (jqXHR, textStatus, errorThrown) {
        var errorResult = {};
        errorResult.toString = function () {
          return errorThrown;
        };
        errorResult.textStatus = textStatus;
        errorResult.jqXHR = jqXHR;
        reject(errorResult); //make sure the toString is not just Object
      };

      //Add the CSRF token
      var token = csrfUtil.getCsrfToken();
      if (!params.headers) {
        params.headers = {
          'x-niagara-csrfToken': token
        };
      }
      else {
        params.headers['x-niagara-csrfToken'] = token;
      }

      $.ajax(uri, params);

    });
  };

  return BackupCommand;
})
;
