/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Patrick Sager
 */

/**
 * API Status: **Private**
 * @module nmodule/ldap/rc/KeytabEditor
 */
define(['baja!',
        'baja!baja:OrdList',
        'bajaux/events',
        'bajaux/mixin/batchSaveMixin',
        'jquery',
        'Promise',
        'underscore',
        'nmodule/webEditors/rc/fe/fe',
        'nmodule/webEditors/rc/fe/baja/BaseEditor',
        'nmodule/webEditors/rc/fe/baja/OrderedMapEditor'], function (
         baja,
         types,
         events,
         batchSaveMixin,
         $,
         Promise,
         _,
         fe,
         BaseEditor,
         OrderedMapEditor) {

  'use strict';

  /**
   * Field editor for changing the selected keytab file on a `ldap:KerberosConfig`
   *
   * @class
   * @alias module:nmodule/webEditors/rc/wb/util/KeytabEditor
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   */
  var KeytabEditor = function KeytabEditor() {
    BaseEditor.apply(this, arguments);
    batchSaveMixin(this);
  };
  KeytabEditor.prototype = Object.create(BaseEditor.prototype);
  KeytabEditor.prototype.constructor = KeytabEditor;

  /**
   * Get the `OrderedMapEditor` from which the user will select a keytab file
   *
   * @private
   * @returns {module:nmodule/webEditors/rc/fe/baja/OrderedMapEditor}
   */
  KeytabEditor.prototype.$getOrderedMapEditor = function () {
    return this.jq().children('.map').data('widget');
  };

  /**
   * Make an empty OrderedMapEditor, we will populate it on load because we need
   * the parent of the BKeytabFile being edited.
   *
   * @param {jQuery} dom
   * @returns {Promise} promise to be resolved after the `OrderedMapEditor`
   * is built
   */
  KeytabEditor.prototype.doInitialize = function (dom) {
    var that = this;

    dom.on(events.MODIFY_EVENT, '.editor', function (e, ed) {
      that.setModified(true);
      return false;
    });

    return fe.buildFor({
      dom: $('<div class="map"/>').appendTo(dom),
      type: OrderedMapEditor,
      value: new baja.OrderedMap()
    });
  };

  /**
   * Populate the ordered map editor with the available keytab files and
   * set the selected value.
   *
   * @param {baja.Struct} value a `ldap:KeytabFile`
   * @returns {Promise} to be resolved once the editor is loaded.
   */
  KeytabEditor.prototype.doLoad = function (value) {
    var that = this;

    return value.getParent().getKeytabsAction()
      .then(function (ordList) {
        var ords = ordList.toString().split("\n");
        var map = that.$getOrderedMapEditor().value();
        _.each(ords, function (ord) {
          map.put(ord, baja.Ord.make(ord));
        });
        return that.$getOrderedMapEditor().load(map, {
          selectedKey: String(that.$value.getKeyTabLocation())
        });
      });
  };

  /**
   * Read the currently selected keytab file location and set the
   * location to the KeytabFile
   *
   * @returns {Promise} promise to be resolved with the selected ldap:KeytabFile
   */
  KeytabEditor.prototype.doRead = function () {
    return this.$getOrderedMapEditor().read()
      .then(function (value) {
        var keytab = baja.$("ldap:KeytabFile");
        keytab.setKeyTabLocation(value);
        return keytab;
      });
  };

  /**
   * Saves the currently selected keytab file location to the KeytabFile Struct
   *
   * @returns {Promise}
   */
  KeytabEditor.prototype.doSave = function (validatedValue, params) {
    var that = this,
      progressCallback = params && params.progressCallback;
    return this.$getOrderedMapEditor().read()
      .then(function (value) {

        var prom = that.value().set({
          slot: "keyTabLocation",
          value: value,
          batch: params && params.batch
        });

        if (progressCallback) {
          progressCallback('commitReady');
        }

        return prom;
      });
  };

  /**
   * Destroy all child editors.
   *
   * @returns {Promise}
   */
  KeytabEditor.prototype.doDestroy = function () {
    return this.getChildEditors().destroyAll();
  };

  /**
   * Enable/disable the dropdown.
   *
   * @param {Boolean} enabled
   * @returns {Promise}
   */
  KeytabEditor.prototype.doEnabled = function (enabled) {
    return this.getChildEditors().setAllEnabled(enabled);
  };

  /**
   * Readonly/writable the dropdown.
   *
   * @param {Boolean} readonly
   * @returns {Promise}
   */
  KeytabEditor.prototype.doReadonly = function (readonly) {
    return this.getChildEditors().setAllReadonly(readonly);
  };

  return KeytabEditor;
});
