/**
 * @copyright 2015, Tridium, Inc. All Rights Reserved.
 * @author Jeremy Narron
 */

/**
 * API Status: **Private**
 * @module nmodule/bql/rc/fe/filter/FilterEntryEditor
 */
define([
  'baja!',
  'baja!bql:FilterEntry',
  'jquery',
  'underscore',
  'Promise',
  'bajaux/events',
  'bajaux/Properties',
  'nmodule/webEditors/rc/fe/baja/ComplexCompositeEditor',
  'bajaux/mixin/subscriberMixIn',
  'css!nmodule/bql/rc/fe/filter/templates/editors'
], function (baja,
             types,
             $, _,
             Promise,
             events,
             Properties,
             ComplexCompositeEditor,
             subscriberMixin) {

  'use strict';

  var LOADED   = events.LOAD_EVENT,
      MODIFIED = events.MODIFY_EVENT;

  /**
   * FilterEntryEditor is a field editor for a `bql:FilterEntry` instance.
   *
   * @class
   * @alias module:nmodule/bql/rc/fe/filter/FilterEntryEditor
   * @extends module:nmodule/webEditors/rc/fe/baja/ComplexCompositeEditor
   */
  var FilterEntryEditor = function () {
    ComplexCompositeEditor.apply(this, arguments);
    subscriberMixin(this);
  };
  FilterEntryEditor.prototype = Object.create(ComplexCompositeEditor.prototype);
  FilterEntryEditor.prototype.constructor = FilterEntryEditor;

  /**
   * Edit the "active" and "filter" slots on a `bql:FilterEntry`.
   *
   * @returns {Array}
   */
  FilterEntryEditor.prototype.getSlotFilter = function () {
    return ['active', 'filter'];
  };

  /**
   * Adds event handlers for enabling/disabling the "filter" slot
   * editor based on the value of the "active" slot.
   *
   * @param dom
   * @returns {Promise}
   */
  FilterEntryEditor.prototype.doInitialize = function (dom) {
    var that = this;

    dom.on([LOADED, MODIFIED].join(' '), '.key-active', function () {
      that.getBuilder().getEditorFor('active').read().then(function (enabled) {
        return that.getBuilder().getEditorFor('filter').setEnabled(enabled);
      })
        .catch(baja.error);
    });

    return ComplexCompositeEditor.prototype.doInitialize.apply(that, arguments);
  };

  /**
   * Sets the "label" slot value as the true/false text for the
   * "active" slot editor.
   *
   * @param key
   * @returns {module:nmodule/webEditors/rc/fe/config/ComplexCompositeBuilder}
   */
  FilterEntryEditor.prototype.makeBuilder = function () {
    var builder      = ComplexCompositeEditor.prototype.makeBuilder.apply(this, arguments),
        getConfigFor = builder.getConfigFor;

    builder.getConfigFor = function (key) {
      return getConfigFor.apply(this, arguments)
        .then(function (config) {
          switch (key) {
            case 'active':
            {
              var filterEntry = builder.getDataSource(),
                  label       = filterEntry.get('label');
              config.properties = Properties.extend(
                config.properties, {trueText: label, falseText: label}
              );
            }
          }
          return config;
        });
    };

    return builder;
  };

  /**
   * Override CompositeEditor#doRead() to return a bql:FilterEntry instance configured with
   * the current active state and filter values.
   *
   * Note: This override is necessary since CompositeEditor#doRead() expects it's component editors
   * to return ComplexDiffs in their doRead() methods. However currently, the individual BQL filter
   * editors return filter instances instead of diffs in their doRead() implementations. This method
   * can be removed once the individual BQL filter editors are refactored to be ComplexCompositeEditors
   *
   * @returns {Promise} Returns a promise resolved with a bql:FilterEntry instance
   */
  FilterEntryEditor.prototype.doRead = function () {
    var builder  = this.getBuilder(),
        label    = this.value().getLabel(),
        activeEd = builder.getEditorFor('active'),
        filterEd = builder.getEditorFor('filter');

    return Promise.join(activeEd.read(), filterEd.read(), function (active, filter) {
      return baja.$('bql:FilterEntry', {active: active, label: label, filter: filter});
    });
  };

  return FilterEntryEditor;
});
