/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author JJ Frankovich
 */

/**
 * API Status: **Private**
 * @module nmodule/webChart/rc/line/DataLayer
 */
define([ "jquery",
  "baja!",
  "d3",
  "nmodule/webChart/rc/model/modelUtil",
  "nmodule/webChart/rc/webChartUtil",
  "nmodule/webChart/rc/chartEvents"
], function (
  $,
  baja,
  d3,
  modelUtil,
  webChartUtil,
  events) {
  "use strict";

  /**
   * @class
   * @alias module:nmodule/webChart/rc/line/DataLayer
   * @param graph
   */
  var DataLayer = function (graph) {
    var that = this;

    that.$graph = graph;
  };

  DataLayer.prototype.initialize = function () {
    var chart = this.$graph.chartd3(),
      jq = this.$graph.widget().jq();

    jq.on(events.SERIES_FOCUS, function (event, series, plotSize) {
      plotSize = plotSize || 0;
      // Make any fixed data pop up text highlight if the data is selected.
      chart.selectAll('.lineContainer .dataSet path')
        .style("stroke-width", function (d) {
          if (plotSize <= 1) {
            return null;
          }
          return series === d ? 3 : 1;
        });
    });

    jq.on(events.SERIES_BLUR, function () {
      // Make any fixed data pop up text highlight if the data is selected.
      chart.selectAll('.lineContainer .dataSet path')
        .style("stroke-width", function () {
          return null;
        });
    });
  };

  /**
   * Get the Fill Opacity for a particular path.
   * @param {webChart.Line} graph
   * @param {Number} pathIndex
   * @return {Number} between 0 and 1 (%)
   */
  DataLayer.prototype.getFillOpacity = function (series, pathIndex) {
    var that = this,
      result,
      length,
      seriesList;

    if (!series.isEnabled()) {
      return 0;
    }
    if (series.isShade()) {

      if (series.isBoolean()) {
        if (pathIndex === 0) {
          result = 0.2;
        } else {
          result = 0.7;
        }
      } else if (series.isDiscrete()) {
        length = modelUtil.getEnumRangeLength(series); //add 1 so that ordinal 0 is not opacity zero (that is skip data).
        result = 0.7 / length * (pathIndex + 1);

      } else {
        //Numeric Shade gets split into 5 categories
        length = 5;
        result = 0.7 / length * (pathIndex + 1);
      }
      seriesList = that.$graph.data();
      return (modelUtil.hasLine(seriesList) || modelUtil.hasBar(seriesList)) ? result * 2 / 3 : result; //reduce opacity if a line is also present
    }
    return null;
  };

  DataLayer.prototype.graphData = function (values) {
    var that = this,
      graph = that.$graph,
      lineSelection,
      areaSelection,
      barSelection,
      lineDataSet,
      areaDataSet,
      barDataSet;

    // One dataSet is a representation of an object containing a name, facets, and points
    // Create the data selection and separate it into its component parts.
    lineSelection = graph.chartd3()
      .select('.lines')
      .selectAll('.dataSet')
      .data(modelUtil.getLines(values),
        function (d) {
          return d.ord();
        });

    areaSelection = graph.chartd3()
      .select('.areas')
      .selectAll('.dataSet')
      .data(modelUtil.getAreas(values),
        function (d) {
          return d.ord();
        });

    barSelection = graph.chartd3()
      .select('.bars')
      .selectAll('.dataSet')
      .data(modelUtil.getBars(values),
        function (d) {
          return d.ord();
        });

    lineDataSet = lineSelection.enter()
      .append('g')
      .attr('class', 'dataSet');

    lineDataSet
      .insert('path');

    areaDataSet = areaSelection.enter()
      .append('g')
      .attr('class', 'dataSet');

    barDataSet = barSelection.enter()
      .append('g')
      .attr('class', 'dataSet');

    areaDataSet
      .each(function (d, i) {
        if (!d) {
          return;
        }

        var select = d3.select(this),
          j;
        select
          .insert('path');
        if (d.isBoolean()) {
          select.insert('path');
        } else if (d.isDiscrete()) {
          for (j = 1; j < modelUtil.getEnumRangeLength(d); j++) {
            select.insert('path');
          }
        } else {
          for (j = 1; j < 5; j++) { //5 total categories for mapping non-discrete to discrete
            select.insert('path');
          }
        }
      });

    function configureSelection(selection) {

      // Give each line a color.
      selection.selectAll('path')
        .attr('class', function (d) {
          if (d.isShade()) {
            return 'area';
          } else {
            return 'line';
          }
        })

        .style('fill', function (d) {
          if (d.isShade()) {
            return that.fillColor(d);
          } else {
            return null;
          }
        })
        .style('fill-opacity', function (d, i) {
          if (d.isShade()) {
            return that.getFillOpacity(d, i);
          } else {
            return null;
          }
        })
        .style('stroke', function (d) {
          if (d.isShade()) {
            return null;
          } else {
            return d.color();
          }
        });

      selection.exit().remove();
    }

    configureSelection(lineSelection);
    configureSelection(areaSelection);
    barSelection.exit().remove();
  };

  DataLayer.prototype.redraw = function () {
    var that = this,
      graph = that.$graph,
      seriesList = graph.data(),
      widget = graph.widget(),
      model = widget.model(),
      settings = widget.settings(),
      barIndex = 0,
      statusBarIndex = 0;

    // Redraw the lines
    graph.chartd3()
      .selectAll('.lineContainer .dataSet')
      .each(function (series, seriesIndex) {
        //lines and areas
        var pathSelection = d3.select(this).selectAll("path"),
          points = series.samplingPoints(),
          scale = series.valueScale().scale(),
          areaY,
          statusSelection,
          valueMin,
          valueBlock,
          height,
          xScale,
          topArea,
          bottomArea,
          barLayout;

        //shade setup
        if (series.isShade()) {

          areaY = that.getAreaY(series);
          height = widget.dimensions().height;
          xScale = graph.getScaleX();
          topArea = height * areaY[0];
          bottomArea = height * areaY[1];

          //numeric shade settup
          if (!series.isDiscrete()) {
            valueMin = scale.domain()[1];
            valueBlock = (scale.domain()[0] - scale.domain()[1]) / 4.0;
            if (!valueBlock) {
              valueBlock = 1;
            }

            for (var j = 0; j < points.length; j++) {
              if (points[j].discreteY === undefined) {
                points[j].discreteY = parseInt((points[j].y - valueMin) / valueBlock);
              }
            }
          }
        }

        //lines and shades
        pathSelection
          .attr('d', function (data, index) {
            if (!points.length) {
              //instead of return null, return empty path to avoid d3 error: "Error: Problem parsing d=""
              return "M0 0";
            }

            var discrete = series.isDiscrete();

            if (series.isShade()) {
              //areas like enum and boolean
              return d3.svg.area()
                .interpolate("step-after")
                .x(function (d) {
                  return xScale(d.x);
                })
                .y1(function (d, i) {
                  return topArea;
                })
                .y0(function (d, i) {
                  if (discrete && d.y === index) {
                    return bottomArea;
                  } else if (!discrete && d.discreteY === index) {
                    return bottomArea;
                  } else {
                    return topArea;
                  }
                })
                .defined(function (d) {
                  return !d.skip;
                })(points);
            } else { //lines
              return d3.svg.line()
                .interpolate(series.getLineInterpolation())
                .x(function (d) {
                  return graph.getScaleX()(d.x);
                })
                .y(function (d, i) {
                  return scale(d.y);
                })
                .defined(function (d) {
                  return !d.skip;
                })(points);
            }
          })
          .style("stroke-opacity", function (d) {
            return d.isEnabled() ? 1 : 0;
          });

        //bars
        if (series.isBar()) {
          height = widget.dimensions().height;
          var barSelection = d3.select(this).selectAll("line.bar").data(points),
            y2 = scale(0),
            color = series.color();

          if (!barLayout) {
            barLayout = modelUtil.getBarLayout(model);
          }

          barSelection.enter()
            .append("line")
            .attr("class", "bar");

          barSelection.exit().remove();

          barSelection
            .style("stroke", color)
            .style("opacity", function (point) {
              return series.isEnabled() ? null : 0;
            })
            .style("stroke-width", function (d) {
              return barLayout.barWidth(d.x);
            })
            .attr("y1", function (d) {
              var y1 = scale(d.y);

              //ensure that barGraphs have a minimum height of 4 pixels to help orientate users.
              if (Math.abs(y2 - y1) < 4) {
                if (y2 - y1 < 0) {
                  y1 += 4;
                } else {
                  y1 -= 4; //slightly negative means to add extra below the line.
                }
              }

              return y1;
            })
            .attr("y2", y2)
            .each(function (point) {

              var elem = d3.select(this),
                x = modelUtil.getBarX(point.x, barIndex, barLayout, graph.getScaleX());

              elem.attr({
                x1: x,
                x2: x
              });


            });
          barIndex++;
        }
        //statusColoring
        if (settings.getStatusColoring() === 'off' ||
          (series.isShade() && modelUtil.hasLine(seriesList) && settings.getBackgroundAreaColor() === "off")) {
          points = [];
        }

        if (series.isShade()) {
          //status area
          var y = widget.dimensions().height * areaY[0] + 6;

          statusSelection = d3.select(this).selectAll("line.status").data(points);

          statusSelection.enter()
            .append("line")
            .attr("class", "status");

          statusSelection.exit().remove();

          statusSelection
            .attr("y1", y)
            .attr("y2", y)
            .style("stroke-opacity", function () {
              return series.isEnabled() ? 1 : 0;
            })
            .style("stroke-width", function (d) {
              return webChartUtil.hasStatusColor(d.status) && !d.skip ? 12 : 0; //about 1em
            })
            .attr("x1", function (d) {
              return graph.getScaleX()(d.x);
            })
            .attr("x2", function (d) {
              var next = points[points.indexOf(d) + 1];
              if (next && !next.skip) {
                return graph.getScaleX()(next.x);
              }
              return graph.getScaleX()(d.x);
            })
            .each(function (d) {
              var select = d3.select(this),
                statusColor = webChartUtil.statusToColor(d.status);
              select.style("stroke", function () {
                var result = statusColor.bgColor;
                if (!result) {
                  result = that.fillColor(series);
                }
                return result;
              });
            });
        } else if (series.isBar()) {
          statusSelection = d3.select(this).selectAll("line.status").data(points);

          statusSelection.enter()
            .append("line")
            .attr("class", "status");

          statusSelection.exit().remove();

          statusSelection
            .style("opacity", function (point) {
              return series.isEnabled() ? null : 0;
            })
            .style("stroke-width", function (point) {
              return webChartUtil.hasStatusColor(point.status) && !point.skip ? barLayout.barWidth(point.x) : 0;
            })
            .each(function (point) {
              //grouping positioning offset
              var elem = d3.select(this),
                x = modelUtil.getBarX(point.x, statusBarIndex, barLayout, graph.getScaleX()), //new
                y1 = scale(point.y),
                y2;

              if (point.y >= 0) {
                y2 = Math.min(scale(0), y1 + 12);
              } else {
                y2 = Math.max(scale(0), y1 - 12);
              }

              elem.attr({
                x1: x,
                x2: x,
                y1: y1,
                y2: y2
              });

              //coloring
              var select = d3.select(this),
                statusColor = webChartUtil.statusToColor(point.status);

              select.style("stroke", function () {
                var result = statusColor.bgColor;
                if (!result) {
                  result = that.fillColor(series);
                }
                return result;
              });
            });
          statusBarIndex++;
        } else {
          //status line
          statusSelection = d3.select(this).selectAll("circle").data(points);
          statusSelection.enter()
            .append("circle")
            .attr("class", "dot")
            .attr("r", 2.75);

          statusSelection.exit().remove();

          statusSelection
            .style("fill-opacity", function () {
              return series.isEnabled() ? 1 : 0;
            })
            .attr("cx", function (d) {
              return graph.getScaleX()(d.x);
            })
            .attr("cy", function (d) {
              return scale(d.y);
            }).each(function (d) {

            var select = d3.select(this),
              statusColor = webChartUtil.statusToColor(d.status);

            select.style("fill", function () {
              var result = statusColor.bgColor;
              if (!result) {
                result = that.fillColor(series);
              }

              return result;
            });
          });
        }
      });
  };

  /**
   * Get the top and bottom percent of the height
   * @param {BaseSeries} series
   * @returns {Array.<Number>}
   */
  DataLayer.prototype.getAreaY = function (series) {
    var that = this,
      graph = that.$graph,
      i,
      seriesList = graph.data(),
      discreteCount = 0,
      discreteIndex = -1;

    for (i = 0; i < seriesList.length; i++) {
      if (seriesList[i].ord() === series.ord()) {
        discreteIndex = discreteCount;
        discreteCount++;
      } else if (seriesList[i].isShade()) {
        discreteCount++;
      }
    }
    return [ discreteIndex / discreteCount, (discreteIndex + 1) / discreteCount ];
  };

  /**
   * Control the Fill for a Color. If discrete, should use line color unless actual lines are present, then use
   * grey.
   * @param {BaseSeries} d
   * @returns {String}
   */
  DataLayer.prototype.fillColor = function (d) {
    var that = this,
      widget = that.$graph.widget();
    if (d.isShade()) {
      if (modelUtil.hasLine(that.$graph.data()) && widget.settings().getBackgroundAreaColor() === "off") {
        return "#c1c1c1"; //background area grey
      }
    }
    return d.color();
  };

  return DataLayer;
});
