/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 * @author Andy Sutton
 */

/**
 * API Status: **Private**
 * @module nmodule/ndriver/rc/util/rpcUtil
 */
define([
    'baja!'
  ], function (
    baja
  ) {

  'use strict';

  /**
   * A utility class for making and dealing with ndriver rpc calls
   *
   * API Status: **Private**
   * @exports nmodule/ndriver/rc/util/rpcUtil
   */
  var exports = {};

  var DEVICE_TYPE_SPEC = 'dv',
    DEVICE_FOLDER_TYPE_SPEC = 'df',
    POINT_FOLDER_TYPE_SPEC = 'pf',
    PROXY_EXT_TYPE_SPEC = 'pe',
    SUBSCRIPTION_DEPTH = 'sd',
    DISCOVERY_LEAF_TYPE_SPEC = 'dl';


  var ndriverRpc = function (methodName, argList, batch) {
    return baja.rpc({
      typeSpec: 'ndriver:NDriverRpcUtil',
      methodName: methodName,
      args: argList,
      batch: batch
    });
  };

  /**
   * Retrieve information required to set up a device manager on this nNetwork.
   * @param {String} nNetworkTypeSpec
   * @returns {Promise.<Object>}
   */
  exports.getDeviceManagerInfo = function (nNetworkTypeSpec) {
    return ndriverRpc('fetchDeviceManagerInfo', [ nNetworkTypeSpec ]);
  };

  /**
   * Retrieve information required to set up a point manager on this nPointDeviceExt/nNetwork.
   * @param {String} nPointDeviceExtTypeSpec
   * @returns {Promise.<Object>}
   */
  exports.getPointManagerInfo = function (nPointDeviceExtTypeSpec) {
    return ndriverRpc('fetchPointManagerInfo', [ nPointDeviceExtTypeSpec ]);
  };

  /**
   * Valid database types for the discovery object.
   * @param {baja.Complex} discovery
   * @returns {Promise.<Array.<String>>}
   */
  exports.getValidDatabaseTypes = function (discovery) {
    return ndriverRpc('getValidDatabaseTypes', [ baja.bson.encodeValue(discovery) ]);
  };

  /**
   * The default Component instance for the discovery object.
   * @param {baja.Complex} discovery
   * @param {String} proxyExtTypeSpec
   * @returns {Promise.<baja.Component>}
   */
  exports.getDefaultDiscoveryInstance = function (discovery, proxyExtTypeSpec) {
    return ndriverRpc('getDefaultDiscoveryInstance', [ baja.bson.encodeValue(discovery), proxyExtTypeSpec ])
      .then(function (encodedComp) {
        return baja.bson.decodeAsync(JSON.parse(encodedComp));
      })
      ;
  };

  /**
   * A Component instance for the discovery object of the given typeSpec.
   * @param {baja.Complex} discovery
   * @param {String} nNetworkTypeSpec
   * @param {String} proxyExtTypeSpec
   * @returns {Promise.<baja.Component>}
   */
  exports.getDiscoveryInstance = function (discovery, typeSpec, proxyExtTypeSpec) {
    return ndriverRpc('getDiscoveryInstance', [ baja.bson.encodeValue(discovery), typeSpec, proxyExtTypeSpec ])
      .then(function (encodedComp) {
        return baja.bson.decodeAsync(JSON.parse(encodedComp));
      });
  };

  /**
   * The default name for the discovery object.
   * @param {baja.Complex} discovery
   * @returns {Promise.<String>}
   */
  exports.getDiscoveryName = function (discovery) {
    return ndriverRpc('getDiscoveryName', [ baja.bson.encodeValue(discovery) ]);
  };

  /**
   * Whether the discovery object exists in the database table for the componentOrd.
   * @param {String} componentOrd
   * @param {baja.Complex} discovery
   * @param {baja.comm.Batch} batch to use for the network calls
   * @returns {Promise.<Boolean>}
   */
  exports.getExisting = function (discovery, componentOrd, batch) {
    return ndriverRpc('getExisting', [ baja.bson.encodeValue(discovery), componentOrd ], batch);
  };

  /**
   * Whether the discovery object exists in the database table for the componentOrd.
   * @param {String} componentOrd
   * @param {baja.Complex} discovery
   * @returns {Promise.<baja.Icon>}
   */
  exports.getDiscoveryIcon = function (discovery) {
    return ndriverRpc('getDiscoveryIcon', [ baja.bson.encodeValue(discovery) ])
      .then(function (encodedIcon) {
        return baja.bson.decodeAsync(JSON.parse(encodedIcon));
      });
  };

  /**
   * Find the Device TypeSpec in the result object.
   * @param {Object} rpcResult
   * @returns {String}
   */
  exports.getDeviceTypeSpecFromResult = function (rpcResult) {
    return rpcResult[DEVICE_TYPE_SPEC];
  };

  /**
   * Find the DeviceFolder Spec in the result object.
   * @param {Object} rpcResult
   * @returns {String}
   */
  exports.getDeviceFolderTypeSpecFromResult = function (rpcResult) {
    return rpcResult[DEVICE_FOLDER_TYPE_SPEC];
  };

  /**
   * Find the PointFolder TypeSpec in the result object.
   * @param {Object} rpcResult
   * @returns {String}
   */
  exports.getPointFolderTypeSpecFromResult = function (rpcResult) {
    return rpcResult[POINT_FOLDER_TYPE_SPEC];
  };

  /**
   * Find the ProxyExt TypeSpec in the result object.
   * @param {Object} rpcResult
   * @returns {String}
   */
  exports.getProxyExtTypeSpecFromResult = function (rpcResult) {
    return rpcResult[PROXY_EXT_TYPE_SPEC];
  };

  /**
   * Find the Subscription Depth in the result object.
   * @param {Object} rpcResult
   * @returns {String}
   */
  exports.getSubscriptionDepthFromResult = function (rpcResult) {
    return rpcResult[SUBSCRIPTION_DEPTH];
  };

  /**
   * Find the DiscoveryLeaf TypeSpec in the result object.
   * @param {Object} rpcResult
   * @returns {String}
   */
  exports.getDiscoveryLeafTypeSpecFromResult = function (rpcResult) {
    return rpcResult[DISCOVERY_LEAF_TYPE_SPEC];
  };

  return exports;
});
