




/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 * @author Andy Sutton
 */

/**
 * API Status: **Private**
 * @module nmodule/ndriver/rc/util/util
 */
define('nmodule/ndriver/rc/util/util',[
    'baja!',
    'baja!ndriver:NNetwork,ndriver:NDevice,driver:PointDeviceExt',
    'lex!',
    'nmodule/webEditors/rc/fe/baja/util/compUtils',
    'nmodule/webEditors/rc/wb/table/model/Column'
  ], function (
    baja,
    types,
    lex,
    compUtils
  ) {

  'use strict';

  var closest = compUtils.closest;

  /**
   * A general purpose utility class for ndriver
   *
   * API Status: **Private**
   * @exports nmodule/ndriver/rc/util/util
   */
  var exports = {};

  /**
   * Returns the lexicon value for the moduleName/lexiconKey, returns blank if not found
   *
   * @param {String} moduleName
   * @param {String} lexiconKey
   */
  exports.getLexiconForModule = function (moduleName, lexiconKey) {
    return lex.module(moduleName)
    .then(function (moduleLex) {
      return  moduleLex.get(lexiconKey);
    })
    .catch(function (ignore) {
      return '';
    });
  };

  /**
   * Find the NDevice for the passed in component.
   *
   * @param {baja.Component} component
   * @returns {baja.Component}
   */
  exports.getNDevice = function (component) {
    return closest(component, 'ndriver:NDevice');
  };

  /**
   * Find the NNetwork for the passed in component.
   *
   * @param {baja.Component} component
   * @returns {baja.Component}
   */
  exports.getNNetwork = function (component) {
    return closest(component, 'ndriver:NNetwork');
  };

  /**
   * Find the NPointDeviceExt for the passed in component.
   *
   * @param {baja.Component} component
   * @returns {baja.Component}
   */
  exports.getNPointDeviceExt = function (component) {
    // Because BNPointDeviceExt implements BINDiscoveryHost (meaning the standard
    // ndriver pointDevExt assumes discovery is supported), for drivers that don't
    // support point discovery the NDriver Wizard creates a pointDeviceExt that
    // extends BPointDeviceExt (and not BNPointDevExt), so we actually need to
    // look for a BPointDeviceExt here.
    return closest(component, 'driver:PointDeviceExt');
  };

  return exports;
});

/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 * @author Andy Sutton
 */

/**
 * API Status: **Private**
 * @module nmodule/ndriver/rc/util/rpcUtil
 */
define('nmodule/ndriver/rc/util/rpcUtil',[
    'baja!'
  ], function (
    baja
  ) {

  'use strict';

  /**
   * A utility class for making and dealing with ndriver rpc calls
   *
   * API Status: **Private**
   * @exports nmodule/ndriver/rc/util/rpcUtil
   */
  var exports = {};

  var DEVICE_TYPE_SPEC = 'dv',
    DEVICE_FOLDER_TYPE_SPEC = 'df',
    POINT_FOLDER_TYPE_SPEC = 'pf',
    PROXY_EXT_TYPE_SPEC = 'pe',
    SUBSCRIPTION_DEPTH = 'sd',
    DISCOVERY_LEAF_TYPE_SPEC = 'dl';


  var ndriverRpc = function (methodName, argList, batch) {
    return baja.rpc({
      typeSpec: 'ndriver:NDriverRpcUtil',
      methodName: methodName,
      args: argList,
      batch: batch
    });
  };

  /**
   * Retrieve information required to set up a device manager on this nNetwork.
   * @param {String} nNetworkTypeSpec
   * @returns {Promise.<Object>}
   */
  exports.getDeviceManagerInfo = function (nNetworkTypeSpec) {
    return ndriverRpc('fetchDeviceManagerInfo', [ nNetworkTypeSpec ]);
  };

  /**
   * Retrieve information required to set up a point manager on this nPointDeviceExt/nNetwork.
   * @param {String} nPointDeviceExtTypeSpec
   * @returns {Promise.<Object>}
   */
  exports.getPointManagerInfo = function (nPointDeviceExtTypeSpec) {
    return ndriverRpc('fetchPointManagerInfo', [ nPointDeviceExtTypeSpec ]);
  };

  /**
   * Valid database types for the discovery object.
   * @param {baja.Complex} discovery
   * @returns {Promise.<Array.<String>>}
   */
  exports.getValidDatabaseTypes = function (discovery) {
    return ndriverRpc('getValidDatabaseTypes', [ baja.bson.encodeValue(discovery) ]);
  };

  /**
   * The default Component instance for the discovery object.
   * @param {baja.Complex} discovery
   * @param {String} proxyExtTypeSpec
   * @returns {Promise.<baja.Component>}
   */
  exports.getDefaultDiscoveryInstance = function (discovery, proxyExtTypeSpec) {
    return ndriverRpc('getDefaultDiscoveryInstance', [ baja.bson.encodeValue(discovery), proxyExtTypeSpec ])
      .then(function (encodedComp) {
        return baja.bson.decodeAsync(JSON.parse(encodedComp));
      })
      ;
  };

  /**
   * A Component instance for the discovery object of the given typeSpec.
   * @param {baja.Complex} discovery
   * @param {String} nNetworkTypeSpec
   * @param {String} proxyExtTypeSpec
   * @returns {Promise.<baja.Component>}
   */
  exports.getDiscoveryInstance = function (discovery, typeSpec, proxyExtTypeSpec) {
    return ndriverRpc('getDiscoveryInstance', [ baja.bson.encodeValue(discovery), typeSpec, proxyExtTypeSpec ])
      .then(function (encodedComp) {
        return baja.bson.decodeAsync(JSON.parse(encodedComp));
      });
  };

  /**
   * The default name for the discovery object.
   * @param {baja.Complex} discovery
   * @returns {Promise.<String>}
   */
  exports.getDiscoveryName = function (discovery) {
    return ndriverRpc('getDiscoveryName', [ baja.bson.encodeValue(discovery) ]);
  };

  /**
   * Whether the discovery object exists in the database table for the componentOrd.
   * @param {String} componentOrd
   * @param {baja.Complex} discovery
   * @param {baja.comm.Batch} batch to use for the network calls
   * @returns {Promise.<Boolean>}
   */
  exports.getExisting = function (discovery, componentOrd, batch) {
    return ndriverRpc('getExisting', [ baja.bson.encodeValue(discovery), componentOrd ], batch);
  };

  /**
   * Whether the discovery object exists in the database table for the componentOrd.
   * @param {String} componentOrd
   * @param {baja.Complex} discovery
   * @returns {Promise.<baja.Icon>}
   */
  exports.getDiscoveryIcon = function (discovery) {
    return ndriverRpc('getDiscoveryIcon', [ baja.bson.encodeValue(discovery) ])
      .then(function (encodedIcon) {
        return baja.bson.decodeAsync(JSON.parse(encodedIcon));
      });
  };

  /**
   * Find the Device TypeSpec in the result object.
   * @param {Object} rpcResult
   * @returns {String}
   */
  exports.getDeviceTypeSpecFromResult = function (rpcResult) {
    return rpcResult[DEVICE_TYPE_SPEC];
  };

  /**
   * Find the DeviceFolder Spec in the result object.
   * @param {Object} rpcResult
   * @returns {String}
   */
  exports.getDeviceFolderTypeSpecFromResult = function (rpcResult) {
    return rpcResult[DEVICE_FOLDER_TYPE_SPEC];
  };

  /**
   * Find the PointFolder TypeSpec in the result object.
   * @param {Object} rpcResult
   * @returns {String}
   */
  exports.getPointFolderTypeSpecFromResult = function (rpcResult) {
    return rpcResult[POINT_FOLDER_TYPE_SPEC];
  };

  /**
   * Find the ProxyExt TypeSpec in the result object.
   * @param {Object} rpcResult
   * @returns {String}
   */
  exports.getProxyExtTypeSpecFromResult = function (rpcResult) {
    return rpcResult[PROXY_EXT_TYPE_SPEC];
  };

  /**
   * Find the Subscription Depth in the result object.
   * @param {Object} rpcResult
   * @returns {String}
   */
  exports.getSubscriptionDepthFromResult = function (rpcResult) {
    return rpcResult[SUBSCRIPTION_DEPTH];
  };

  /**
   * Find the DiscoveryLeaf TypeSpec in the result object.
   * @param {Object} rpcResult
   * @returns {String}
   */
  exports.getDiscoveryLeafTypeSpecFromResult = function (rpcResult) {
    return rpcResult[DISCOVERY_LEAF_TYPE_SPEC];
  };

  return exports;
});

/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 * @author Andy Sutton
 */

/**
 * API Status: **Private**
 * @module nmodule/ndriver/rc/column/DiscoveryPropertyMgrColumn
 */
define('nmodule/ndriver/rc/column/DiscoveryPropertyMgrColumn',[
    'baja!',
    'Promise',
    'underscore',
    'nmodule/ndriver/rc/util/util',
    'nmodule/webEditors/rc/wb/mgr/model/columns/PropertyPathMgrColumn'
  ], function (
     baja,
     Promise,
     _,
     util,
     PropertyPathMgrColumn
  ) {

  'use strict';

  /**
   * PropertyMgrColumn subclass for use with ndriver discovery.
   *
   * @class
   * @alias module:nmodule/ndriver/rc/column/DiscoveryPropertyMgrColumn
   * @extends module:nmodule/webEditors/rc/wb/mgr/model/columns/PropertyMgrColumn
   * @param {String} name A unique name for this column
   * @param {String} path A slot path, specified as a '/' delimited string
   * @param {Object} params
   */
  var DiscoveryPropertyMgrColumn = function DiscoveryPropertyMgrColumn(name, path, params) {
    PropertyPathMgrColumn.apply(this, arguments);

    this.$moduleName = (params && params.moduleName) || '';
    this.$columnIndex = (params && params.columnIndex) || -1;
  };

  DiscoveryPropertyMgrColumn.prototype = Object.create(PropertyPathMgrColumn.prototype);
  DiscoveryPropertyMgrColumn.prototype.constructor = DiscoveryPropertyMgrColumn;

  /**
   * Returns a diplayName for this column.
   *
   * @returns {Promise.<String>}
   */
  DiscoveryPropertyMgrColumn.prototype.toDisplayName = function () {
    var that = this;

    // 1. if a displayName has been supplied as a parameter, use that.
    if (that.$displayName) {
      return Promise.resolve(that.$displayName);
    }

    return Promise.resolve(that.$moduleName)
    .then(function (moduleNameLexLookup) {
      // 2. try looking up the module's lexicon
      return moduleNameLexLookup ? util.getLexiconForModule(moduleNameLexLookup, that.$name) : undefined;
    })
    .then(function (moduleLexResult) {
      // 3. return the result if found or try the webEditors module lexicon
      return moduleLexResult || util.getLexiconForModule('webEditors', that.$name);
    })
    .then(function (lexLookupResult) {
      // 4. return the result if found or last resort, the call the superclass implementation
      return lexLookupResult || PropertyPathMgrColumn.prototype.toDisplayName.apply(that);
    });
  };

  function isGroup(row) {
    return row.getTreeNode().isGroup && row.getTreeNode().isGroup();
  }

  /**
   * Returns the row's value for this column
   *
   * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row
   * @returns {String}
   */
  DiscoveryPropertyMgrColumn.prototype.getValueFor = function (row) {

    if (isGroup(row)) {
      // Show the row description in the first column after the icon
      return this.$columnIndex === 1 ? row.getSubject().getDescription() : '';
    }
    return PropertyPathMgrColumn.prototype.getValueFor.apply(this, arguments);
  };

  return DiscoveryPropertyMgrColumn;
});

/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 * @author Andy Sutton
 */

/**
 * API Status: **Private**
 * @module nmodule/ndriver/rc/column/PasswordPropertyMgrColumn
 */
define('nmodule/ndriver/rc/column/PasswordPropertyMgrColumn',[
    'baja!',
    'Promise',
    'underscore',
    'nmodule/webEditors/rc/wb/mgr/model/columns/PropertyPathMgrColumn',
    'nmodule/webEditors/rc/wb/mgr/model/MgrColumn',
    'nmodule/webEditors/rc/servlets/password',
    'nmodule/webEditors/rc/fe/baja/util/typeUtils',
    'nmodule/webEditors/rc/fe/baja/util/compUtils'
  ], function (
    baja,
    Promise,
    _,
    PropertyPathMgrColumn,
    MgrColumn,
    password,
    typeUtils,
    compUtils
  ) {

  'use strict';

  var getParentComponent = compUtils.getParentComponent;

  /**
   * PropertyMgrColumn subclass for use with ndriver.
   *
   * @class
   * @alias module:nmodule/ndriver/rc/column/PasswordPropertyMgrColumn
   * @extends module:nmodule/webEditors/rc/wb/mgr/model/columns/PropertyPathMgrColumn
   * @param {String} name Optional A name for this column (if not provided, will be calculated)
   * @param {String} path A slot path, specified as a '/' delimited string
   * @param {Object} params
   */
  var PasswordPropertyMgrColumn = function PasswordPropertyMgrColumn(name, path, params) {
    PropertyPathMgrColumn.apply(this, arguments);
  };

  PasswordPropertyMgrColumn.prototype = Object.create(PropertyPathMgrColumn.prototype);
  PasswordPropertyMgrColumn.prototype.constructor = PasswordPropertyMgrColumn;

  /**
   * Recursively set all all properties from value on target, adding if not present on target,
   * using the password servlet to set any passwords.
   *
   * @param {baja.Component} target
   * @param {baja.Complex} value
   * @param {baja.comm.Batch} batch
   * @returns {Promise}
   */
  function doSet(target, value, batch) {
    var sets = [];

    value.getSlots().properties().each(function (slot) {
      if (slot.getType().isComplex()) {
        var targetValue = target.get(slot.getName());
        if (targetValue) {
          sets.push(doSet(targetValue, value.get(slot), batch));
        } else {
          sets.push(
            compUtils.writeSlot(target, slot.getName(), value.get(slot).newCopy(true), batch)
          );
        }
      } else if (slot.getType().is('baja:Password')) {
        sets.push(
          password.setPassword(
            value.get(slot).encodeToString(),
            slot.getName(),
            target,
            batch
          )
        );
      } else {
        sets.push(
          compUtils.writeSlot(target, slot.getName(), value.get(slot), batch)
        );
      }
    });
    return Promise.all(sets);
  }

  /**
   * When committing changes to a password, take one of two paths
   * depending on whether we're editing an existing one, or creating a new one.
   *
   * If editing an existing password, the change will be delegated out to the
   * 'password' servlet.
   *
   * For a new password, we go ahead and set the slot value.
   * The password will be wiped when writing it up to the station;
   * only after the password is mounted will the password itself
   * be sent to the servlet. See NMgrModel#addInstances().
   *
   * @param {baja.Value} value
   * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row
   * @param {Object} [params]
   * @returns {Promise}
   */
  PasswordPropertyMgrColumn.prototype.commit = function (value, row, params) {
    var name = this.getPropertyName(),
      propValue = this.getComplexFromPath(row),
      progressCallback = params && params.progressCallback,
      batch = params && params.batch;

    function finish(prom) {
      if (progressCallback) { progressCallback(MgrColumn.COMMIT_READY); }
      return prom;
    }

    var parent = getParentComponent(propValue);

    if (parent && parent.isMounted()) {
      var promises = [];
      if (value.getType().is('baja:Password')) {
        promises.push(password.setPassword(
            value.encodeToString(),
            name,
            propValue,
            batch
          )
        );
      } else {
        promises.push(doSet(propValue.get(name), value, batch));
      }

      return finish(Promise.all(promises));
    }

    if (value.getType().is('baja:Password')) {
      // For an unmounted password, ie on a brand new instance, write the password value,
      // and it will get dealt with in NMgrModel#addInstances().
      propValue.set({ slot: name, value: value.encodeToString() });

      return finish(Promise.resolve());
    }

    // For other unmounted types, ie on a brand new instance, defer to the super class.
    return PropertyPathMgrColumn.prototype.commit.apply(this, arguments);
  };

  return (PasswordPropertyMgrColumn);
});

/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 * @author Andy Sutton
 */

/**
 * API Status: **Private**
 * @module nmodule/ndriver/rc/util/columnUtil
 */
define('nmodule/ndriver/rc/util/columnUtil',[
    'baja!',
    'underscore',
    'Promise',
    'nmodule/webEditors/rc/fe/fe',
    'nmodule/webEditors/rc/wb/table/model/Column',
    'nmodule/webEditors/rc/wb/PropertySheet',
    'nmodule/webEditors/rc/wb/mgr/model/columns/PropertyPathMgrColumn',
    'nmodule/ndriver/rc/column/DiscoveryPropertyMgrColumn',
    'nmodule/ndriver/rc/column/PasswordPropertyMgrColumn'
  ], function (
    baja,
    _,
    Promise,
    fe,
    Column,
    PropertySheet,
    PropertyPathMgrColumn,
    DiscoveryPropertyMgrColumn,
    PasswordPropertyMgrColumn
  ) {

  'use strict';

  var PROXY_EXT = 'driver:ProxyExt',
    KEY_MGR = 'Mgr',
    MGR_EDIT = 'ed',
    MGR_UNSEEN = 'un',
    MGR_READONLY = 'ro',
    UNSEEN = Column.flags.UNSEEN,
    EDITABLE = Column.flags.EDITABLE,
    READONLY = Column.flags.READONLY;


  // Check if specified property's default value uses a custom field editor
  // see com.tridium.ndriver.ui.NMgrColumnUtil.java#hasCustomFE for the equivalent wb implementation
  var hasCustomFE = function (property) {

    return fe.getDefaultConstructor(property.getType(), {
      formFactors: [ 'compact', 'mini' ]
    })
    .then(function (fieldEditor) {
      if (fieldEditor && !(fieldEditor instanceof PropertySheet)) {
         return true;
      }
      return false;
    });
  };

  /**
   * returns a path in dot notation form
   * @param {String} path
   *
   * @returns {String}
   */
  var pathToDotNotation = function (path) {
    // Convert the path to dot notation format, then remove any leading 'proxyExt.'
    // (This helps for the point manager where most columns are prop paths under
    // a control point starting with the control point's proxyExt property)
    // This is equivalent to initial processing in NMgrColumnUtil#getMgrColumnName.

    return path.replace(/\//g, '.').replace(/^proxyExt\./, '');
  };

  var usePasswordColumn = function (property) {
    var returnValue = false;

    if (property.getType().is('baja:Password')) {
      returnValue = true;
    } else if (property.getType().isComplex()) {
      property.$getValue().getSlots().properties().each(function (slot) {
        if (usePasswordColumn(slot)) { returnValue = true; }
      });
    }
    return returnValue;
  };

  /**
   * make an ndriver column.
   *
   * @param {boolean} isDiscovery - true if creating for a discovery table.
   * @param {Array.<String>} path - the property path array
   * @param {baja.Property} property - the property the column is for
   * @param {Object} params - the params object for the column
   *
   * @returns {Promise.<Array.<module:nmodule/webEditors/rc/wb/table/model/Column>>}
   */
  var makeColumn = function (isDiscovery, path, property, params) {
    if (isDiscovery) {
      return new DiscoveryPropertyMgrColumn(pathToDotNotation(path), path, params);
    }

    // deal with passwords ...
    if (usePasswordColumn(property)) {
      return new PasswordPropertyMgrColumn(pathToDotNotation(path), path, params);
    }

    return new PropertyPathMgrColumn(pathToDotNotation(path), path, params);
  };

  var addColumn = function (complex, columns, slotParentType, path, isDiscovery, property, mgrEdit) {
    var flags,
      params = {};

    if (mgrEdit.indexOf(MGR_EDIT) > -1) { flags |= EDITABLE; }
    if (mgrEdit.indexOf(MGR_UNSEEN) > -1) { flags |= UNSEEN; }
    if (mgrEdit.indexOf(MGR_READONLY) > -1) { flags |= READONLY; }

    if (flags) { params.flags = flags; }

    params.moduleName = complex.getType().getModuleName();

    // For a frozen property, set the Type instance in order to resolve its display name/facets
    //  ( see nmodule/webEditors/rc/wb/table/model/columns/PropertyColumn [params.type] )
    if (slotParentType && property.isFrozen()) { params.type = slotParentType; }

    if (isDiscovery) { params.columnIndex = columns.length; }

    columns.push(
      makeColumn(isDiscovery, path + property.getName(), property, params)
    );
  };

  var recurse = function (complex, columnsIn, slotParentType, path, isDiscovery) {
    var properties = complex.getSlots().properties().toArray();

    if (properties.length === 0) { return Promise.resolve(columnsIn); }

    return properties.reduce(function (resolveColumns, property, index) {
      return Promise.all([
        property.getType().isSimple(),
        property.getType().isStruct(),
        hasCustomFE(property),
        property.getType().isComplex(),
        resolveColumns
      ]).spread(function (isSimple, isStruct, hasCustomFE,  isComplex, columns) {
        var mgrEdit = property.getFacets().get(KEY_MGR);

        // not using 'if(!mgrEdit)' here because an empty string means MGR is present as a Facet
        // but not set to ReadOnly, Unseen or Editable.
        var hasMgrEditFacet = typeof mgrEdit === 'string';

        if (hasMgrEditFacet) {
          if (isSimple || (isStruct && hasCustomFE)) {
            addColumn(complex, columns, slotParentType, path, isDiscovery, property, mgrEdit);

          } else if (isComplex) { // not sure if it's possible to end up with an object that isn't a
            // complex here, so the 'else if' could potentially just be 'else'
            // but it's been left because this is what NMgrColumnUtil.java#recurse does.
            return recurse(property.$getValue(), columns, property.$getValue().getType(), path + property.getName() + '/', isDiscovery);
          }
        }
        return columns;
      });
    }, columnsIn);

  };

  /**
   * A utility class for dealing with ndriver columns
   *
   * API Status: **Private**
   * @exports nmodule/ndriver/rc/util/columnUtil
   */
  var exports = {};

  /*
   * Get the columns for a database or discovery table.
   *
   * @param  {String} complex - the object to create columns from.
   * @param  {Array.<module:nmodule/webEditors/rc/wb/table/model/Column>} columns
   *                         - the current list of columns to append to.
   * @param  {boolean} isDiscovery - true if creating for a discovery table.
   * @returns {Promise.<Array.<module:nmodule/webEditors/rc/wb/table/model/Column>>}
   */
  exports.getColumnsFor = function (complex, columns, isDiscovery) {
    return recurse(complex, columns,
      complex.getType(),    // slot parent type
      baja.hasType(complex, PROXY_EXT) ? 'proxyExt/' : '', // path
      isDiscovery);
  };

  return exports;
});

/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 * @author Andy Sutton
 */

/**
 * API Status: **Private**
 * @module nmodule/ndriver/rc/util/learnUtil
 */
define('nmodule/ndriver/rc/util/learnUtil',[
    'baja!',
    'baja!' +
    'control:NumericWritable,control:NumericPoint,control:BooleanWritable,control:BooleanPoint,' +
    'control:EnumWritable,control:EnumPoint,control:StringWritable,control:StringPoint,' +
    'ndriver:NDiscoveryGroup,ndriver:NDeviceManager',
    'underscore',
    'Promise',
    'jquery',
    'nmodule/ndriver/rc/util/util',
    'nmodule/ndriver/rc/util/columnUtil',
    'nmodule/ndriver/rc/util/rpcUtil',
    'nmodule/webEditors/rc/wb/table/tree/TreeTableModel',
    'nmodule/webEditors/rc/wb/tree/TreeNode',
    'nmodule/webEditors/rc/wb/mgr/model/columns/IconMgrColumn'
  ], function (
    baja,
    types,
    _,
    Promise,
    $,
    util,
    columnUtil,
    rpcUtil,
    TreeTableModel,
    TreeNode,
    IconMgrColumn
  ) {

  'use strict';

  var N_DISCOVERY_GROUP = 'ndriver:NDiscoveryGroup',
    I_N_DISCOVERY_ICON = 'ndriver:INDiscoveryIcon',

    DEVICE_ICON = baja.Icon.make([ 'module://icons/x16/device.png' ]),
    DEVICE_FOLDER_ICON = baja.Icon.make([ 'module://icons/x16/deviceFolder.png' ]),
    POINT_FOLDER_ICON = baja.Icon.make([ 'module://icons/x16/pointFolder.png' ]),

    NUMERIC_WRITABLE_ICON = baja.Icon.make([ 'module://icons/x16/control/numericPoint.png' ]),
    BOOLEAN_WRITABLE_ICON = baja.Icon.make([ 'module://icons/x16/control/booleanPoint.png' ]),
    ENUM_WRITABLE_ICON = baja.Icon.make([ 'module://icons/x16/control/enumPoint.png' ]),
    STRING_WRITABLE_ICON = baja.Icon.make([ 'module://icons/x16/control/stringPoint.png' ]),

    NUMERIC_POINT_ICON = baja.Icon.make([ 'module://icons/x16/statusNumeric.png' ]),
    BOOLEAN_POINT_ICON = baja.Icon.make([ 'module://icons/x16/statusBoolean.png' ]),
    ENUM_POINT_ICON = baja.Icon.make([ 'module://icons/x16/statusEnum.png' ]),
    STRING_POINT_ICON = baja.Icon.make([ 'module://icons/x16/statusString.png' ]),

    DEFAULT_POINT_ICON = null;


  var getDiscoveryIcon = function (discovery, isPointManager, isDiscoveryGroup) {
    var defaultTypeSpec, defaultType,
      discoveryType = discovery.getType();

    if (discoveryType.is(I_N_DISCOVERY_ICON)) {
      return rpcUtil.getDiscoveryIcon(discovery);
    }

    if (!isPointManager) {
      return isDiscoveryGroup ? DEVICE_FOLDER_ICON : DEVICE_ICON;
    }

    // at this point we must have a point manager

    if (isDiscoveryGroup) {
      return POINT_FOLDER_ICON;
    }


    // TODO - although this next bit works, it makes a network call for each discovery object.
    //            Might want to rethink how this is done and/or how important the icons are.
    //            (the same applies to the rpcUtil.getDiscoveryIcon call above).

    // For discovered points, try to match the discovered type with one of the ControlPoint types
    // see com.tridium.ndriver.ui.NMgrLearn.java#getPointIconDefault
    return rpcUtil.getValidDatabaseTypes(discovery)
      .then(function (validTypeSpecs) {
        if (!validTypeSpecs || validTypeSpecs.length === 0) {
          return DEFAULT_POINT_ICON;
        }

        defaultTypeSpec = validTypeSpecs[0];

        if (!defaultTypeSpec) {
          return DEFAULT_POINT_ICON;
        }

        defaultType = baja.$(defaultTypeSpec).getType();

        if (defaultType.is('control:NumericWritable')) { return NUMERIC_WRITABLE_ICON; }
        if (defaultType.is('control:NumericPoint')) { return NUMERIC_POINT_ICON; }
        if (defaultType.is('control:BooleanWritable')) { return BOOLEAN_WRITABLE_ICON; }
        if (defaultType.is('control:BooleanPoint')) { return BOOLEAN_POINT_ICON; }
        if (defaultType.is('control:EnumWritable')) { return ENUM_WRITABLE_ICON; }
        if (defaultType.is('control:EnumPoint')) { return ENUM_POINT_ICON; }
        if (defaultType.is('control:StringWritable')) { return STRING_WRITABLE_ICON; }
        if (defaultType.is('control:StringPoint')) { return STRING_POINT_ICON; }

        return DEFAULT_POINT_ICON;
      });
  };


  /**
   * A utility class for dealing with ndriver discovery
   *
   * API Status: **Private**
   * @exports nmodule/ndriver/rc/util/learnUtil
   */
  var exports = {};

  /*
   * Make a tree node for an item in the discovery table.
   *
   * @param  {baja.Complex} discovery - a discovered item.
   * @param  {String} discoveryLeafTypeSpec - the type spec of the discoveryLeaf.
   * @param  {Boolean} isPointManager - whether the manager is a PointManager, (false assumes it's a DeviceManager).
   * @returns {Promise.<module:nmodule/webEditors/rc/wb/tree/TreeNode>}
   */
  exports.makeDiscoveryTableNode = function (discovery, discoveryLeafTypeSpec, isPointManager) {
    var name = discovery.getName(),
      displayName = discovery.getDisplayName(),
      mayHaveKids = _.constant(false),
      kids = [],
      promisesForKids = [],
      isGroup = _.constant(false),
      isDiscoveryGroup = discovery.getType().is(N_DISCOVERY_GROUP); // using NDiscoveryGroup rather than
                                                                    // INDiscoveryGroup as it has a 'description' property

    if (isDiscoveryGroup) {
      displayName = discovery.getDescription();

      mayHaveKids = _.constant(true);
      isGroup = _.constant(true);

      _.each(discovery.getSlots().toValueArray(), function (kid) {
        if (kid.getType().is(discoveryLeafTypeSpec) || baja.hasType(kid, N_DISCOVERY_GROUP)) {
          promisesForKids.push(
            exports.makeDiscoveryTableNode(kid, discoveryLeafTypeSpec, isPointManager)
          );
        }
      });
    }


    return Promise.all(promisesForKids)
      .then(function (resolvedKids) {
        kids = resolvedKids;
        return getDiscoveryIcon(discovery, isPointManager, isDiscoveryGroup);
      })
      .then(function (icon) {
        var node = new TreeNode(name, displayName, kids);
        node.value = _.constant(discovery);
        node.getIcon = _.constant(icon);
        node.mayHaveKids = mayHaveKids;
        node.isGroup = isGroup;

        return node;
      });
  };

  /*
   * Make the model for the discovery table.
   *
   * @param  {String} discoveryLeafTypeSpec - the type spec of the discoveryLeaf.
   * @returns {Promise.<module:nmodule/webEditors/rc/wb/table/tree/TreeTableModel>}
   */
  exports.makeLearnModel = function (discoveryLeafTypeSpec) {
    return columnUtil.getColumnsFor(
      baja.$(discoveryLeafTypeSpec),
      [ new IconMgrColumn() ], // standard columns
      true                     // 'isDicovery'
    ).then(function (columns) {
      return TreeTableModel.make({
        columns: columns
      });
    });
  };

  return exports;
});

/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 * @author Andy Sutton
 */

/**
 * A mixin that provides the shared bajaux Manager View functionality for NDriver
 *
 * API Status: **Private**
 * @module nmodule/ndriver/rc/Automanager
 */
define('nmodule/ndriver/rc/Automanager',[
    'baja!',
    'baja!ndriver:NDiscoveryGroup,baja:Password',
    'log!nmodule.ndriver.rc.Automanager',
    'Promise',
    'underscore',
    'jquery',
    'bajaux/Widget',
    'nmodule/webEditors/rc/wb/mgr/Manager',
    'bajaux/util/CommandButtonGroup',
    'nmodule/webEditors/rc/fe/fe',
    'nmodule/ndriver/rc/util/util',
    'nmodule/ndriver/rc/util/rpcUtil',
    'nmodule/ndriver/rc/util/learnUtil',
    'nmodule/driver/rc/wb/mgr/DriverMgr',
    'nmodule/driver/rc/wb/mgr/PointMgr',
    'nmodule/webEditors/rc/fe/baja/util/DepthSubscriber',
    'nmodule/webEditors/rc/wb/mgr/MgrTypeInfo',
    'nmodule/webEditors/rc/wb/mixin/mixinUtils',
    'bajaux/commands/CommandGroup',
    'nmodule/webEditors/rc/wb/mgr/MgrLearn',
    'nmodule/webEditors/rc/wb/mgr/MgrLearnTableSupport',
    'bajaScript/baja/obj/Password',
    'nmodule/webEditors/rc/servlets/password'
  ], function (
    baja,
    types,
    log,
    Promise,
    _,
    $,
    Widget,
    Manager,
    CommandButtonGroup,
    fe,
    util,
    rpcUtil,
    learnUtil,
    DriverMgr,
    PointMgr,
    DepthSubscriber,
    MgrTypeInfo,
    mixinUtils,
    CommandGroup,
    addLearnSupport,
    MgrLearnTableSupport,
    Password,
    passwordServlet
  ) {

  'use strict';

  var applyMixin = mixinUtils.applyMixin,
    hasMixin = mixinUtils.hasMixin,
    logError = log.severe.bind(log),
    N_DISCOVERY_GROUP = 'ndriver:NDiscoveryGroup',
    MIXIN_NAME = 'AUTOMANAGER';

  /*
   * Workbench and UX have differing definitions of subscription depth, so an
   * adjustment is required.
   *
   *  BComponent.java#lease:
   *    If depth is greater than zero then the lease includes descendants
   *    (one is children, two is children and grandchildren, etc).
   *  DepthSubscripber.js#subscribeAll
   *     @param {Number} depth the depth to subscribe - 0 subscribes nothing,
   *     1 subscribes only the given components, 2 subscribes components
   *     and their kids, etc.
   */
  var UX_SUBSCRIPTION_DEPTH_ADJ = 1;

  /*
   * A default subscription depth for the discovery job.
   * 3 caters for BNDiscoveryGroup folder within a BNDiscoveryGroup folder
   */
  var DEFAULT_DISCOVERY_JOB_SUBSCRIPTION_DEPTH = 3;

  /**
   * A mixin to provide Automanager support to ndriver manager views.
   *
   * @alias module:nmodule/ndriver/rc/Automanager
   * @mixin
   * @extends module:nmodule/driver/rc/wb/mgr/DriverMgr
   * @param {module:nmodule/driver/rc/wb/mgr/DriverMgr} target
   * @param {Object} params
   *
   */
  var Automanager = function Automanager(target, params) {

    if (!(target instanceof DriverMgr)) {
      throw new Error('target must be a DriverMgr instance.');
    }

    if (!applyMixin(target, MIXIN_NAME, Automanager.prototype)) {
      return this;
    }

    var superDoInitialize = target.doInitialize,
      superLoad = target.load;

    /**
     * Extend the base Manager's doInitialize() function,
     * adding the jobcomplete handler.
     *
     * @override
     * @param {JQuery} dom
     * @returns {*|Promise}
     */
    target.doInitialize = function (dom, params) {
      var that = this;

      that.on('jobcomplete', function (job) {
        that.$updateLearnTableModelFromJob(job).catch(logError);
      });

      return superDoInitialize.call(that, dom, {});
    };

    /**
     * Extend the base Manager's load() function, looking up/populating automanager properties
     * and adding discovery if required.
     *
     * @override
     * @param {baja.Component} component
     * @returns {Promise}
     */
    target.load = function (component) {
      var that = this, args = arguments;

      return that.setUpManagerInfo(component)
      .then(function (managerInfo) {

        return Promise.all([
          that.$setupForSubscriptionDepth(managerInfo, component),
          that.$setupForDiscovery(managerInfo)
        ]);
      }).then(function () {
        // make sure the types we need are available locally
        return baja.importTypes({
          typeSpecs: that.getTypesToImport()
        });
      }).then(function () {
        return superLoad.apply(that, args);
      });
    };
  };

  /**
   * update this manager's subscriber with the depth configured in BNNetwork.
   *
   * @private
   * @param {Object} managerInfo
   * @param {baja.Component} component
   * @returns {Promise}
   */
  Automanager.prototype.$setupForSubscriptionDepth = function (managerInfo, component) {
    var subscriptionDepth = rpcUtil.getSubscriptionDepthFromResult(managerInfo) + UX_SUBSCRIPTION_DEPTH_ADJ;

    if (!subscriptionDepth || subscriptionDepth === this.getSubscriber().getDepth()) {
      return Promise.resolve();
    }

    this.$subscriber = new DepthSubscriber(subscriptionDepth);

    return this.getSubscriber().subscribe(component);
  };

  /**
   * add discovery setup, if this manager supports discovery.
   *
   * @private
   * @param {Object} managerInfo
   * @returns {Promise}
   */
  Automanager.prototype.$setupForDiscovery = function (managerInfo) {
    var that = this;

    that.$discoveryLeafTypeSpec = rpcUtil.getDiscoveryLeafTypeSpecFromResult(managerInfo);

    if (!that.isDiscoverySupported()) { return Promise.resolve(); }

    // discovery is supported ...
    function filterCommands(cmd) {
      return (hasMixin(cmd, 'MGR_COMMAND')) ? cmd.isShownInActionBar() : true;
    }

    // save the command container as it's already been initialized
    var commandContainerElement = that.jq().find('.commandContainer').detach();

    // replace the standard manager html with the Learn Table template
    that.jq().html(MgrLearnTableSupport.mgrLearnHtml());

    // add the mgr-action-bar class to the standard element, then slot it into the new html
    commandContainerElement.addClass('mgr-action-bar');
    that.jq().find('.commandContainer.mgr-action-bar').replaceWith(commandContainerElement);

    addLearnSupport(that);

    that.makeDiscoveryCommands().forEach(function (discoveryCommand) {
      that.getCommandGroup().add(discoveryCommand);
    });

    /**
     * Extend the base Manager's getExisting() function,
     * delegating the proccessing to the server (and ultimately this ndriver's java
     * implementation of isExisting) via an rpc call.
     *
     * @override
     * @param {baja.Complex} discovery a discovery item
     * @param {baja.comm.Batch} [batch] batch to use for the target `getExisting`
     * @returns {Promise.<baja.Component|undefined>} the existing component that was found to match
     * the given discovery node, or undefined if no such match was found.
     */
    that.getExisting = function (discovery, batch) {
      return rpcUtil.getExisting(discovery, this.value().getNavOrd().toString(), batch);
    };

    return that.initializeJobBar(that.jq())
    .then(function () {
      return Widget.in(that.$getCommandContainerElement().children('.CommandButtonGroup')[1]).load(
        that.getCommandGroup().filter({ include: filterCommands })
      );
    });
  };

  /**
   * An array of type specs that the manager implementation needs
   *
   * @return {Array.<String>} An array of typeSpecs strings
   */
  Automanager.prototype.getTypesToImport = function () {
    var types = this.getManagerTypes();
    if (this.isDiscoverySupported()) {
      types.push(this.getDiscoveryLeafTypeSpec());
    }
    return types;
  };

  /**
   * The typeSpec for the device used by this manager
   *
   * @return {String} the typeSpec for the device
   */
  Automanager.prototype.getDeviceTypeSpec = function () {
    return this.$deviceTypeSpec;
  };

  /**
   * Setup information required by this manager, usually acquired by means of an rpc call to populate
   * values from the ndriver/Automanager interfaces
   *
   * @abstract
   * @param {baja.Component} component - the value being loaded into the manager.
   * @return {Object} key/value pairs of manager info
   */
  Automanager.prototype.setUpManagerInfo = function (component) {
    throw new Error('setUpManagerInfo() function not implemented.');
  };

  /**
   * An array of type specs that this manager implementation needs
   *
   * @abstract
   * @return {Array.<String>}
   */
  Automanager.prototype.getManagerTypes = function () {
    throw new Error('getManagerTypes() function not implemented.');
  };

  /**
   * Provides ndriver automanager specific functionality for adding instances to manager views.
   *
   * @param {Array.<baja.Component>} instances the instances to add
   * @param {Array.<String>} names the desired slot names for the instances.
   * @param {Function} superAddInstances the addInstances function from the super class.
   * @param {module:nmodule/webEditors/rc/wb/mgr/model/MgrModel} mgrModel the base ord for instance being added.
   *
   * @returns {Promise} promise to be resolved when the instances are added.
   */
  Automanager.prototype.addInstancesForNDriver = function (instances, names, superAddInstances, mgrModel) {
    var that = this,
      baseNavOrd = that.value().getNavOrd(),
      passwordChanges = [],
      defaultPassword = Password.DEFAULT;


    var processPasswordSlot = function (slot, parent, baseNavOrd) {
      var pw = parent.get(slot);

      if (!pw.equals(defaultPassword)) {
        // store the details of required password changes for later use

        passwordChanges.push({
          pw: pw.encodeToString(),
          slotName: slot.getName(),
          base: baja.Ord.make(baseNavOrd)
        });
      }

      // instance is unmounted so set will be a sync call.
      parent.set({
        slot: slot,
        value: defaultPassword
      }).catch(logError);
    };

    var processPropertySlots = function (component, baseNavOrd) {
      baseNavOrd = baseNavOrd + '/' + component.getName();
      return component.getSlots().properties().each(function (slot) {
        if (slot.getType().is('baja:Password')) {
          processPasswordSlot(slot, component, baseNavOrd);
        } else if (slot.getType().isComplex() || slot.getType().isStruct()) {
          // if we have a complex or a struct, carry on down the tree
          processPropertySlots(component.get(slot), baseNavOrd);
        }
      });
    };

    // check each instance for slots that can't be saved on unmounted components,
    // specifically ones with passwords
    instances.forEach(function (instance) {
      processPropertySlots(instance, baseNavOrd);
    });

    return superAddInstances.apply(mgrModel, arguments)
      .then(function () {
        // once the parent component is mounted, we can make the password changes
        var batch = new baja.comm.Batch();
        return batch.commit(
          passwordChanges.map(function (passwordChange) {
            return passwordServlet.setPassword(passwordChange.pw, passwordChange.slotName, passwordChange.base, { batch: batch });
          })
        );
      });
  };




  // Discovery ...

  /**
   * The typeSpec for the discovery leaf used by this manager.
   *
   * @return {String}
   */
  Automanager.prototype.getDiscoveryLeafTypeSpec = function () {
    return this.$discoveryLeafTypeSpec;
  };

  /**
   * Whether this manager supports discovery
   *
   * @return {boolean}
   */
  Automanager.prototype.isDiscoverySupported = function () {
    //if getDiscoveryLeafTypeSpec() is falsey it will be taken to mean that discovery is not supported
    return !!this.getDiscoveryLeafTypeSpec();
  };

  /**
   * return the component that has the discovery interface implementations for this manager
   * (the NNetwork for the device manager, or the NPointDeviceExt for the point manager)
   *
   * @return {baja.Component}
   */
  Automanager.prototype.getDiscoveryComponent = function () {
    if (this.isDiscoverySupported()) {
      // only needed if discovery is supported
      throw new Error('getDiscoveryComponent() function not implemented.');
    }
  };

  /**
   * Make the model for the discovery table.
   *
   * @override
   * @function module:webEditors/rc/wb/mgr/MgrLearn#makeLearnModel

   * @returns {Promise.<module:nmodule/webEditors/rc/wb/table/tree/TreeTableModel>}
   */
  Automanager.prototype.makeLearnModel = function () {
    return learnUtil.makeLearnModel(this.getDiscoveryLeafTypeSpec());
  };

  /**
   * Invoke an Action on the station that will submit a discovery job, then
   * set the returned ORD on the manager.
   *
   * @override
   * @function module:webEditors/rc/wb/mgr/MgrLearn#doDiscover
   *
   * @returns {Promise}
   */
  Automanager.prototype.doDiscover = function () {
    var that = this,
      discoveryComp = that.getDiscoveryComponent();

    return Promise.all([
      discoveryComp.submitDiscoveryJob(discoveryComp.getDiscoveryPreferences()),
      that.getDiscoveryJobSubscriptionDepth()
    ])
    .spread(function (ord, depth) {
      ord = baja.Ord.make({
        base: baja.Ord.make('station:'),
        child: ord.relativizeToSession()
      });
      return that.setJob({
        jobOrOrd: ord,
        depth: depth
      });
    });
  };

  /**
   * Return the type(s) suitable for the given discovery item. Some managers may
   * need to inspect the discovery value to return a suitable type or several
   * types.
   *
   * @override
   * @function module:webEditors/rc/wb/mgr/MgrLearn#getTypesForDiscoverySubject
   *
   * @param {baja.Value} discoveredObject
   * @returns {Promise.<module:nmodule/webEditors/rc/wb/mgr/MgrTypeInfo>}
   */
  Automanager.prototype.getTypesForDiscoverySubject = function (discoveredObject) {
    return rpcUtil.getValidDatabaseTypes(discoveredObject)
    .then(function (types) {
      return MgrTypeInfo.make(types);
    });
  };

  /**
   * The subscription depth for the discovery job.
   *
   * @return {Promise.<Number>}
   */
  Automanager.prototype.getDiscoveryJobSubscriptionDepth = function () {
    return Promise.resolve(DEFAULT_DISCOVERY_JOB_SUBSCRIPTION_DEPTH);
  };

  /**
   * Called asynchronously after the job submitted by doDiscover() has
   * finished. This should get the items found in the discovery and
   * update the TreeNodes in the learn table.
   *
   * @private
   * @param {baja.Component} job - the config discovery job.
   * @returns {Promise}
   */
  Automanager.prototype.$updateLearnTableModelFromJob = function (job) {
    var that = this;

    return job.loadSlots()
    .then(function () {
      var discoveries = job.getSlots()
        .is(N_DISCOVERY_GROUP, that.getDiscoveryLeafTypeSpec())
        .toValueArray();

      that.$discoveries = discoveries;
      return that.$updateLearnTable(discoveries);
    });
  };

  /**
   * Function to update the model for the learn table with the discovered
   * items obtained from the job.
   *
   * @private
   * @param {Array.<baja.Component>} discoveries
   * @returns {Promise}
   */
  Automanager.prototype.$updateLearnTable = function (discoveries) {
    var that = this,
      model = that.getLearnModel(),
      currentCount = model.getRows().length;

    return Promise.resolve(currentCount && model.clearRows())
      .then(function () {
        return Promise.all(
          // Create TreeNodes with a value returning the discovered item.
          _.map(discoveries || [], function (discovery) {
            return learnUtil.makeDiscoveryTableNode(
              discovery,
              that.getDiscoveryLeafTypeSpec(),
              (that instanceof PointMgr)
            );
          })
        );
      })
      .then(function (nodes) {
        // Update the model with the discoveries.
        let promises = _.map(nodes, (node) => model.getRootNode().add(node));
        promises.push(model.insertRows(nodes, 0));

        return Promise.all(promises);
      });
  };

  /**
   * Get the values to be set as the proposals on the batch component editor for
   * the rows being added via the AddCommand.
   *
   * @override
   * @function module:webEditors/rc/wb/mgr/MgrLearn#getProposedValuesFromDiscovery
   *
   * @param {baja.Complex} discovery an item obtained from a node in discovery table.
   * @param {*} subject - the subject of the `Row` whose values are to be proposed.
   * @see module:nmodule/webEditors/rc/wb/table/tree/TreeNode
   * @returns {Object|Promise.<Object>} an object literal with the name and initial values to be used for the new component.
   */
  Automanager.prototype.getProposedValuesFromDiscovery = function (discovery, subject) {
    return rpcUtil.getDiscoveryName(discovery)
    .then(function (discoveryName) {
      return {
        name: baja.SlotPath.unescape(discoveryName),
        values: {}
      };
    });
  };

  /**
   * Calls back to the BNDiscoveryLeaf to set up a new instance for this discovery object.
   *
   * @override
   * @function module:webEditors/rc/wb/mgr/MgrLearn#newInstanceFromDiscoverySubject
   *
   * @param {baja.Complex} discovery an instance of a discovery item (e.g. an
   * `ndriver:NDiscoveryLeaf`), dragged from the discovery table and dropped
   * onto the database table or selected when the 'Add' command was invoked.
   *
   * @param {Array.<module:nmodule/webEditors/rc/wb/mgr/MgrTypeInfo>} typeInfos - an
   * array of MgrTypeInfos, created from the type or types returned
   * by the manager's `getTypesForDiscoverySubject()` implementation.
   *
   * @returns {Promise} a Promise of new component instance for the discovered item.
   *
   */
  Automanager.prototype.newInstanceFromDiscoverySubject = function (discovery, typeInfos) {
    var proxyExtTypeSpec = this.getModel().getProxyExtType ? this.getModel().getProxyExtType() : '';
    return rpcUtil.getDefaultDiscoveryInstance(discovery, proxyExtTypeSpec);
  };

  /**
   * Returns a new instance of for the given typeInfo.
   *
   * If a value is supplied for discovery, an rpc call is made to the BNDiscoveryLeaf
   * and returns a new instance of the supplied typeSpec based on the discovery object
   * (Typically called when the Type is changed in the editor).
   *
   * If no discovery value is supplied, the superNewInstance function is called.
   *
   * @param {module:nmodule/webEditors/rc/wb/mgr/MgrTypeInfo} typeInfo
   * @param {Object} [params]
   * @param {baja.Component} [params.discovery] a discovery object
   * @param {Function} newInstance - a newInstance function to call if no discovery object is supplied
   * @param {module:nmodule/webEditors/rc/wb/mgr/model/MgrModel} model
   *
   * @returns {Promise.<baja.Component>} resolves to a new component instance.
   */
  Automanager.prototype.getNewInstance = function (typeInfo, params, newInstance, model) {
    var discovery = params && params.discovery;

    if (discovery) {
      return rpcUtil.getDiscoveryInstance(
        discovery,
        typeInfo.getType().getTypeSpec().toString(),
        this.getModel().getProxyExtType ? this.getModel().getProxyExtType() : ''
      );
    }

    return newInstance.call(model, typeInfo);
  };

  /**
   * Automanager does not implement #isExisting, despite it being required by
   * MgrLearn#getExisting
   * @see {@link module:nmodule/webEditors/rc/wb/mgr/MgrLearn#getExisting}
   * Instead Automanager overrides #getExisting and delegates processing via an
   * rpc call to the java implementation of #isExisting on the DiscoveryLeaf.
   *
   * @param {baja.Complex} discovery
   * @param {baja.Component} component
   *
   * @returns {boolean}  - true if the discovery item and station component match.
   */
  Automanager.prototype.isExisting = function (discovery, component) {
    throw new Error('isExisting() not supported by Automanager');
  };

  // End Discovery



  return (Automanager);
});

/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 * @author Andy Sutton
 */

/**
 * API Status: **Private**
 * @module nmodule/ndriver/rc/column/NTypeMgrColumn
 */
define('nmodule/ndriver/rc/column/NTypeMgrColumn',[
    'baja!',
    'nmodule/webEditors/rc/wb/mgr/model/columns/TypeMgrColumn'
  ], function (
    baja,
    TypeMgrColumn
  ) {

  'use strict';

  /**
   * An implementation of TypeMgrColumn for ndriver that caters for using a
   * discovery object to configure a newInstance..
   *
   * @class
   * @alias module:nmodule/ndriver/rc/column/NTypeMgrColumn
   * @extends module:nmodule/webEditors/rc/wb/mgr/model/MgrColumn
   * @param {Object} params
   */
  var NTypeMgrColumn = function NTypeMgrColumn(params) {
    TypeMgrColumn.apply(this, arguments);
  };

  NTypeMgrColumn.prototype = Object.create(TypeMgrColumn.prototype);
  NTypeMgrColumn.prototype.constructor = NTypeMgrColumn;

  /**
   * Returns a new instance for the selected typeInfo.
   *
   * @param {module:nmodule/webEditors/rc/wb/mgr/model/MgrModel} mgrModel
   * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row
   * @returns {Promise.<baja.Value>}
   */
  NTypeMgrColumn.prototype.newInstance = function (mgrModel, row) {
    // mgrModel will be an ndriver MgrModel that knows how to
    // deal with the additional params argument.
    return mgrModel.newInstance(
      row.data(TypeMgrColumn.SELECTED_TYPE_KEY), // typeInfo
      { discovery: row.data(TypeMgrColumn.DISCOVERY_DATA_KEY) } // params
    );
  };

  return (NTypeMgrColumn);
});

/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 * @author Andy Sutton
 */

/**
 * API Status: **Private**
 * @module nmodule/ndriver/rc/column/OutMgrColumn
 */
define('nmodule/ndriver/rc/column/OutMgrColumn',[
    'baja!',
    'Promise',
    'underscore',
    'nmodule/ndriver/rc/util/util',
    'nmodule/webEditors/rc/wb/mgr/model/MgrColumn'
  ], function (
     baja,
     Promise,
     _,
     util,
     MgrColumn
  ) {

  'use strict';

  var DEFAULT_COLUMN_NAME = 'Out',
    LEXICON_KEY = 'out';

  /**
   * MgrColumn subclass that allows the out slot to be shown without specifying
   * a type to pick up a displayName.
   *
   * @class
   * @alias module:nmodule/ndriver/rc/column/OutMgrColumn
   * @extends module:nmodule/webEditors/rc/wb/mgr/model/MgrColumn
   * @param {Object} params
   */
  var OutMgrColumn = function OutMgrColumn(params) {
    var that = this;

    MgrColumn.call(that, 'out', params);

    if (params.moduleName) {
      that.$moduleName = params.moduleName;
    }
  };
  OutMgrColumn.prototype = Object.create(MgrColumn.prototype);
  OutMgrColumn.prototype.constructor = OutMgrColumn;

  /**
   * Returns a diplayName for this column.
   *
   * @returns {Promise.<String>}
   */
  OutMgrColumn.prototype.toDisplayName = function () {
    var that = this;

    // 1. if a displayName has been supplied as a parameter, use it.
    if (that.$displayName) { return Promise.resolve(that.$displayName); }

    return Promise.resolve(that.$moduleName)
    .then(function (moduleNameLexLookup) {
      // 2. try looking up the module's lexicon
      return moduleNameLexLookup ? util.getLexiconForModule(moduleNameLexLookup, LEXICON_KEY) : undefined;
    })
    .then(function (moduleLexResult) {
      // 3. return the result if found or try the driver module lexicon
      return moduleLexResult || util.getLexiconForModule('driver', LEXICON_KEY);
    })
    .then(function (driverLexResult) {
      // 4. return the result if found or last resort, the default name
      return driverLexResult || DEFAULT_COLUMN_NAME;
    });
  };

  /**
   * Returns the value represented by this row.
   *
   * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row
   * @returns {*}
   */
  OutMgrColumn.prototype.getValueFor = function (row) {
    return String(row.getSubject());
  };

  return OutMgrColumn;
});



/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 * @author Andy Sutton
 */

/**
 * API Status: **Private**
 * @module nmodule/ndriver/rc/util/modelUtil
 */
define('nmodule/ndriver/rc/util/modelUtil',[
    'baja!',
    'baja!control:ControlPoint',
    'nmodule/ndriver/rc/util/columnUtil',
    'nmodule/webEditors/rc/wb/table/model/Column',
    'nmodule/ndriver/rc/column/OutMgrColumn',
    'nmodule/ndriver/rc/column/NTypeMgrColumn',
    'nmodule/webEditors/rc/wb/mgr/model/columns/IconMgrColumn',
    'nmodule/webEditors/rc/wb/mgr/model/columns/PathMgrColumn',
    'nmodule/webEditors/rc/wb/mgr/model/columns/NameMgrColumn',
    'nmodule/webEditors/rc/wb/mgr/model/columns/PropertyMgrColumn',
    'nmodule/webEditors/rc/wb/mgr/model/columns/PropertyPathMgrColumn',
    'nmodule/driver/rc/wb/mgr/columns/DeviceExtsMgrColumn'
  ], function (
    baja,
    types,
    columnUtil,
    Column,
    OutMgrColumn,
    NTypeMgrColumn,
    IconMgrColumn,
    PathMgrColumn,
    NameMgrColumn,
    PropertyMgrColumn,
    PropertyPathMgrColumn,
    DeviceExtsMgrColumn
  ) {

  'use strict';

                                      // flag descriptions taken from javax.baja.workbench.mgr.MgrColumn
  var UNSEEN = Column.flags.UNSEEN,   // The unseen flag is used on columns which are not shown
                                      //   by default until the user turns them on via table options
    EDITABLE = Column.flags.EDITABLE, // Editable indicates a column which is included for edits
                                      //   via the MgrEdit APIs.
    READONLY = Column.flags.READONLY; // The Readonly flag is used on editable columns which are displayed
                                      //   via the MgrEdit dialog, but not user modifiable

  /**
   * A utility class for dealing with ndriver manager models
   *
   * API Status: **Private**
   * @exports nmodule/ndriver/rc/util/modelUtil
   */
  var exports = {};

  /**
   * Create the appropriate columns for this manager/component
   *
   * @param {module:nmodule/ndriver/rc/Automanager} manager
   * @param {baja.Component} component
   * @returns {Array.<module:nmodule/webEditors/rc/wb/table/model/Column>}
   * @throws {Error}
   */
  exports.makeColumns = function (manager, component) {

    var componentInstance,
        columns = [
          // Icon, Path, Name & Type always apply
          new IconMgrColumn(),
          new PathMgrColumn({ flags: UNSEEN }),
          new NameMgrColumn({ flags: EDITABLE }),
          new NTypeMgrColumn({ flags: EDITABLE })
        ];

    // For device managers, BNNetwork or BNDeviceFolder
    if (baja.hasType(component, 'driver:IDeviceFolder')) {
      if (!manager.getDeviceTypeSpec()) { throw new Error('Manager has no DeviceTypeSpec value'); }

      columns.push(
        new DeviceExtsMgrColumn('exts', {
          component: baja.$(manager.getDeviceTypeSpec())
        })
      );

      // use component's getDeviceType in getColumnsFor
      componentInstance = baja.$(manager.getDeviceTypeSpec());


    // for Point Managers, BNPointFolder or BPointDeviceExt
    } else if (baja.hasType(component, 'driver:IPointFolder')) {
      if (!manager.getProxyExtTypeSpec()) { throw new Error('Manager has no ProxyExtTypeSpec value'); }

      columns.push(
        new OutMgrColumn({
          moduleName: component.getType().getModuleName(),
          flags: READONLY
        }),

        new PropertyMgrColumn('facets', {
          type: baja.lt('control:ControlPoint'),
          flags: UNSEEN | EDITABLE
        })
      );

      componentInstance = baja.$(manager.getProxyExtTypeSpec());
    } else {
      throw new Error('unsupported component: ' + component.getType());
    }

    columns = columnUtil.getColumnsFor(componentInstance, columns);

    return columns;
  };


  return exports;
});

/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 * @author Andy Sutton
 */

/**
 * API Status: **Private**
 * @module nmodule/ndriver/rc/NDeviceManager
 */
define('nmodule/ndriver/rc/NDeviceManager',[
    'baja!',
    'baja!ndriver:NDeviceFolder',
    'Promise',
    'underscore',
    'nmodule/ndriver/rc/Automanager',
    'nmodule/ndriver/rc/util/util',
    'nmodule/ndriver/rc/util/rpcUtil',
    'nmodule/ndriver/rc/util/modelUtil',
    'nmodule/driver/rc/wb/mgr/DeviceMgr',
    'nmodule/webEditors/rc/wb/mgr/MgrTypeInfo',
    'nmodule/driver/rc/wb/mgr/DeviceMgrModel',
    'nmodule/webEditors/rc/wb/mgr/MgrLearn'
  ], function (
    baja,
    types,
    Promise,
    _,
    addAutomanagerSupport,
    util,
    rpcUtil,
    modelUtil,
    DeviceMgr,
    MgrTypeInfo,
    DeviceMgrModel,
    addLearnSupport
  ) {

  'use strict';

  var DEFAULT_SUBSCRIPTION_DEPTH = 2;

  /**
   * A default bajaux Device Manager View for NDriver
   *
   * @class
   * @extends module:nmodule/driver/rc/wb/mgr/DeviceMgr
   * @alias module:nmodule/ndriver/rc/NDeviceManager
   */
  var NDeviceManager = function NDeviceManager(params) {
    var that = this;

    DeviceMgr.call(that, _.extend({
      // moduleName & keyName have to be specified here, otherwise
      // the DriverMgr constructor throws an error.
      // We'll attempt to set them to module specific values later (in #makeModel())

      //[params.moduleName] the module name used for lexicon entries for this view.
      moduleName: 'ndriver',

      // [params.keyName] the key name used for lexicon entries for this view.
      keyName: 'NDeviceManager',

      // folderType is set to 'dummy' value (albeit valid), because the mixin doesn't
      // initialise properly if you don't. Again we'll change it in makeModel()

      // [params.folderType] optional parameter indicating the folder type used for the manager view. This will be used by the NewFolder command.
      folderType: baja.lt('ndriver:NDeviceFolder'),

      //[params.subscriptionDepth] the depth to subscribe the component tree.
      subscriptionDepth: DEFAULT_SUBSCRIPTION_DEPTH
    }, params));

    addAutomanagerSupport(that);
  };

  NDeviceManager.prototype = Object.create(DeviceMgr.prototype);
  NDeviceManager.prototype.constructor = NDeviceManager;


  /**
   * Setup information required by this manager, usually acquired by means of an rpc call to populate
   * values from the ndriver/Automanager interfaces
   *
   * @param {baja.Component} component - the value being loaded into the manager.
   * @return {Object} key/value pairs of manager info
   */
  NDeviceManager.prototype.setUpManagerInfo = function (component) {
    var that = this,
      networkType = util.getNNetwork(component).getType(),
      networkTypeSpec = networkType.getTypeSpec();

    return rpcUtil.getDeviceManagerInfo(networkTypeSpec)
    .then(function (networkInfo) {
      that.$moduleName = networkType.getModuleName();

      that.$deviceTypeSpec = rpcUtil.getDeviceTypeSpecFromResult(networkInfo);
      that.$folderType = rpcUtil.getDeviceFolderTypeSpecFromResult(networkInfo);

      return networkInfo;
    });
  };

  /**
   * An array of type specs that this manager implementation needs
   *
   * @return {Array.<String>}
   */
  NDeviceManager.prototype.getManagerTypes = function () {
    return [
      this.getDeviceTypeSpec(),
      this.getFolderType()
    ];
  };

  /**
   * Make the MgrModel for this Device Manager.
   *
   * @param {baja.Component} component - the NNetwork or NDeviceFolder that the Device Manager will display.
   * @returns {Promise.<module:nmodule/webEditors/rc/wb/mgr/model/MgrModel>}
   */
  NDeviceManager.prototype.makeModel = function (component) {
    var that = this;

    return Promise.all([
      MgrTypeInfo.make({
        from: baja.lt(that.getDeviceTypeSpec()),
        concreteTypes: true
      }),
      modelUtil.makeColumns(that, component)
    ]).spread(function (newTypes, columns) {
      return new DeviceMgrModel({
        component: component,
        newTypes: newTypes,
        folderType: baja.lt(that.getFolderType()),
        columns: columns
      });
    })
    .then(function (model) {
      var superNewInstance = model.newInstance,
        superAddInstances = model.addInstances;

      /**
       * Override the model's newInstance function.
       *
       * @param {module:nmodule/webEditors/rc/wb/mgr/MgrTypeInfo} typeInfo
       * @param {Object} [params]
       * @param {baja.Complex} [params.discovery] a discovery item
       * @returns {Promise.<baja.Component>} resolves to a new component instance.
       */
      model.newInstance = function (typeInfo, params) {
        return that.getNewInstance(typeInfo, params, superNewInstance, this /* the model */);
      };

      /**
       * Override the model's addInstances function.
       *
       * @param {Array.<baja.Component>} instances the instances to add
       * @param {Array.<String>} [names] the desired slot names for the instances.
       * If omitted, default names derived from the instance Types will be used.
       * @returns {Promise} promise to be resolved when the instances are added.
       */
      model.addInstances = function (instances, names) {
        return that.addInstancesForNDriver(instances, names, superAddInstances, this);
      };

      return model;
    });
  };

  /**
   * return the component that has discovery interface implementations for this manager
   *
   * @return {baja.Component}
   */
  NDeviceManager.prototype.getDiscoveryComponent = function () {
    return util.getNNetwork(this.value());
  };

  return NDeviceManager;
});


/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 * @author Andy Sutton
 */

/**
 * API Status: **Private**
 * @module nmodule/ndriver/rc/NPointManager
 */
define('nmodule/ndriver/rc/NPointManager',[
    'baja!',
    'baja!ndriver:NPointFolder',
    'Promise',
    'underscore',
    'nmodule/ndriver/rc/Automanager',
    'nmodule/ndriver/rc/util/util',
    'nmodule/ndriver/rc/util/rpcUtil',
    'nmodule/ndriver/rc/util/modelUtil',
    'nmodule/driver/rc/wb/mgr/PointMgr',
    'nmodule/webEditors/rc/wb/mgr/MgrTypeInfo',
    'nmodule/driver/rc/wb/mgr/PointMgrModel'
  ], function (
    baja,
    types,
    Promise,
    _,
    addAutomanagerSupport,
    util,
    rpcUtil,
    modelUtil,
    PointMgr,
    MgrTypeInfo,
    PointMgrModel
  ) {

  'use strict';

  var DEFAULT_SUBSCRIPTION_DEPTH = 3;

  /**
   * A default bajaux Point Manager View for NDriver
   *
   * @class
   * @extends module:nmodule/driver/rc/wb/mgr/PointMgr
   * @alias module:nmodule/ndriver/rc/NPointManager
   */
  var NPointManager = function NPointManager(params) {
    var that = this;

    PointMgr.call(that, _.extend({
      // moduleName & keyName have to be specified here, otherwise
      // the DriverMgr constructor throws an error.
      // We'll attempt to set them to module specific values later (in makeModel())

      //[params.moduleName] the module name used for lexicon entries for this view.
      moduleName: 'ndriver',

      // [params.keyName] the key name used for lexicon entries for this view.
      keyName: 'NPointManager',

      // folderType is set to 'dummy' value (albeit valid), because the mixin doesn't
      // initialise properly if you don't. Again we'll change it in makeModel()

      // [params.folderType] optional parameter indicating the folder type used for the manager view. This will be used by the NewFolder command.
      folderType: baja.lt('ndriver:NPointFolder'),

      //[params.subscriptionDepth] the depth to subscribe the component tree.
      subscriptionDepth: DEFAULT_SUBSCRIPTION_DEPTH
    }, params));

    addAutomanagerSupport(that);
  };

  NPointManager.prototype = Object.create(PointMgr.prototype);
  NPointManager.prototype.constructor = NPointManager;

  /**
   * Setup information required by this manager, usually acquired by means of an rpc call to populate
   * values from the ndriver/Automanager interfaces
   *
   * @param {baja.Component} component - the value being loaded into the manager.
   * @return {Object} key/value pairs of manager info
   */
  NPointManager.prototype.setUpManagerInfo = function (component) {
    var that = this,
      pointManagerType = util.getNPointDeviceExt(component).getType(),
      pointManagerTypeSpec = pointManagerType.getTypeSpec();

    return rpcUtil.getPointManagerInfo(pointManagerTypeSpec)
      .then(function (pointManagerInfo) {

        that.$deviceTypeSpec = rpcUtil.getDeviceTypeSpecFromResult(pointManagerInfo);
        that.$folderType = rpcUtil.getPointFolderTypeSpecFromResult(pointManagerInfo);
        that.$proxyExtTypeSpec = rpcUtil.getProxyExtTypeSpecFromResult(pointManagerInfo);

        that.$moduleName = pointManagerType.getModuleName();

        return pointManagerInfo;
      });
  };

  /**
   * An array of type specs that this manager implementation needs
   *
   * @abstract
   * @return {Array.<String>} An array of typeSpecs strings
   */
  NPointManager.prototype.getManagerTypes = function () {
    return [
      this.getDeviceTypeSpec(),
      this.getFolderType(),
      this.getProxyExtTypeSpec()
    ];
  };

  /**
   * Make the MgrModel for this Point Manager.
   *
   * @param {baja.Component} component - the NPointDeviceExt or NPointFolder that the Point Manager will display.
   * @returns {Promise.<module:nmodule/webEditors/rc/wb/mgr/model/MgrModel>}
   */
  NPointManager.prototype.makeModel = function (component) {
    var that = this;

    return Promise.all([
      PointMgrModel.getDefaultNewTypes(),
      modelUtil.makeColumns(that, component)
    ]).spread(function (newTypes, columns) {
      return new PointMgrModel({
        component: component,
        newTypes: newTypes,
        folderType: baja.lt(that.getFolderType()),
        proxyExtType: that.getProxyExtTypeSpec(),
        columns: columns
      });
    })
    .then(function (model) {
      var superNewInstance = model.newInstance,
        superAddInstances = model.addInstances;

      /**
       * Override the model's newInstance function.
       *
       * @param {module:nmodule/webEditors/rc/wb/mgr/MgrTypeInfo} typeInfo
       * @param {Object} [params]
       * @param {baja.Complex} [params.discovery] a discovery item
       * @returns {Promise.<baja.Component>} resolves to a new component instance.
       */
      model.newInstance = function (typeInfo, params) {
        return that.getNewInstance(typeInfo, params, superNewInstance, this /* the model */);
      };

      /**
       * Override the model's addInstances function.
       *
       * @param {Array.<baja.Component>} instances the instances to add
       * @param {Array.<String>} [names] the desired slot names for the instances.
       * If omitted, default names derived from the instance Types will be used.
       * @returns {Promise} promise to be resolved when the instances are added.
       */
      model.addInstances = function (instances, names) {
        return that.addInstancesForNDriver(instances, names, superAddInstances, this);
      };


      return model;
    });
  };

  /**
   * return the proxy ext type spce used by this manager
   *
   * @return {String}
   */
  NPointManager.prototype.getProxyExtTypeSpec = function () {
    return this.$proxyExtTypeSpec;
  };

  /**
   * return the component that has discovery interface implementations for this manager
   *
   * @return {baja.Component}
   */
  NPointManager.prototype.getDiscoveryComponent = function () {
    return util.getNPointDeviceExt(this.value());
  };

  return NPointManager;
});

