/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 * @author Andy Sutton
 */

/**
 * API Status: **Private**
 * @module nmodule/ndriver/rc/column/PasswordPropertyMgrColumn
 */
define([
    'baja!',
    'Promise',
    'underscore',
    'nmodule/webEditors/rc/wb/mgr/model/columns/PropertyPathMgrColumn',
    'nmodule/webEditors/rc/wb/mgr/model/MgrColumn',
    'nmodule/webEditors/rc/servlets/password',
    'nmodule/webEditors/rc/fe/baja/util/typeUtils',
    'nmodule/webEditors/rc/fe/baja/util/compUtils'
  ], function (
    baja,
    Promise,
    _,
    PropertyPathMgrColumn,
    MgrColumn,
    password,
    typeUtils,
    compUtils
  ) {

  'use strict';

  var getParentComponent = compUtils.getParentComponent;

  /**
   * PropertyMgrColumn subclass for use with ndriver.
   *
   * @class
   * @alias module:nmodule/ndriver/rc/column/PasswordPropertyMgrColumn
   * @extends module:nmodule/webEditors/rc/wb/mgr/model/columns/PropertyPathMgrColumn
   * @param {String} name Optional A name for this column (if not provided, will be calculated)
   * @param {String} path A slot path, specified as a '/' delimited string
   * @param {Object} params
   */
  var PasswordPropertyMgrColumn = function PasswordPropertyMgrColumn(name, path, params) {
    PropertyPathMgrColumn.apply(this, arguments);
  };

  PasswordPropertyMgrColumn.prototype = Object.create(PropertyPathMgrColumn.prototype);
  PasswordPropertyMgrColumn.prototype.constructor = PasswordPropertyMgrColumn;

  /**
   * Recursively set all all properties from value on target, adding if not present on target,
   * using the password servlet to set any passwords.
   *
   * @param {baja.Component} target
   * @param {baja.Complex} value
   * @param {baja.comm.Batch} batch
   * @returns {Promise}
   */
  function doSet(target, value, batch) {
    var sets = [];

    value.getSlots().properties().each(function (slot) {
      if (slot.getType().isComplex()) {
        var targetValue = target.get(slot.getName());
        if (targetValue) {
          sets.push(doSet(targetValue, value.get(slot), batch));
        } else {
          sets.push(
            compUtils.writeSlot(target, slot.getName(), value.get(slot).newCopy(true), batch)
          );
        }
      } else if (slot.getType().is('baja:Password')) {
        sets.push(
          password.setPassword(
            value.get(slot).encodeToString(),
            slot.getName(),
            target,
            batch
          )
        );
      } else {
        sets.push(
          compUtils.writeSlot(target, slot.getName(), value.get(slot), batch)
        );
      }
    });
    return Promise.all(sets);
  }

  /**
   * When committing changes to a password, take one of two paths
   * depending on whether we're editing an existing one, or creating a new one.
   *
   * If editing an existing password, the change will be delegated out to the
   * 'password' servlet.
   *
   * For a new password, we go ahead and set the slot value.
   * The password will be wiped when writing it up to the station;
   * only after the password is mounted will the password itself
   * be sent to the servlet. See NMgrModel#addInstances().
   *
   * @param {baja.Value} value
   * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row
   * @param {Object} [params]
   * @returns {Promise}
   */
  PasswordPropertyMgrColumn.prototype.commit = function (value, row, params) {
    var name = this.getPropertyName(),
      propValue = this.getComplexFromPath(row),
      progressCallback = params && params.progressCallback,
      batch = params && params.batch;

    function finish(prom) {
      if (progressCallback) { progressCallback(MgrColumn.COMMIT_READY); }
      return prom;
    }

    var parent = getParentComponent(propValue);

    if (parent && parent.isMounted()) {
      var promises = [];
      if (value.getType().is('baja:Password')) {
        promises.push(password.setPassword(
            value.encodeToString(),
            name,
            propValue,
            batch
          )
        );
      } else {
        promises.push(doSet(propValue.get(name), value, batch));
      }

      return finish(Promise.all(promises));
    }

    if (value.getType().is('baja:Password')) {
      // For an unmounted password, ie on a brand new instance, write the password value,
      // and it will get dealt with in NMgrModel#addInstances().
      propValue.set({ slot: name, value: value.encodeToString() });

      return finish(Promise.resolve());
    }

    // For other unmounted types, ie on a brand new instance, defer to the super class.
    return PropertyPathMgrColumn.prototype.commit.apply(this, arguments);
  };

  return (PasswordPropertyMgrColumn);
});
