/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 * @author Andy Sutton
 */

/**
 * A mixin that provides the shared bajaux Manager View functionality for NDriver
 *
 * API Status: **Private**
 * @module nmodule/ndriver/rc/Automanager
 */
define([
    'baja!',
    'baja!ndriver:NDiscoveryGroup,baja:Password',
    'log!nmodule.ndriver.rc.Automanager',
    'Promise',
    'underscore',
    'jquery',
    'bajaux/Widget',
    'nmodule/webEditors/rc/wb/mgr/Manager',
    'bajaux/util/CommandButtonGroup',
    'nmodule/webEditors/rc/fe/fe',
    'nmodule/ndriver/rc/util/util',
    'nmodule/ndriver/rc/util/rpcUtil',
    'nmodule/ndriver/rc/util/learnUtil',
    'nmodule/driver/rc/wb/mgr/DriverMgr',
    'nmodule/driver/rc/wb/mgr/PointMgr',
    'nmodule/webEditors/rc/fe/baja/util/DepthSubscriber',
    'nmodule/webEditors/rc/wb/mgr/MgrTypeInfo',
    'nmodule/webEditors/rc/wb/mixin/mixinUtils',
    'bajaux/commands/CommandGroup',
    'nmodule/webEditors/rc/wb/mgr/MgrLearn',
    'nmodule/webEditors/rc/wb/mgr/MgrLearnTableSupport',
    'bajaScript/baja/obj/Password',
    'nmodule/webEditors/rc/servlets/password'
  ], function (
    baja,
    types,
    log,
    Promise,
    _,
    $,
    Widget,
    Manager,
    CommandButtonGroup,
    fe,
    util,
    rpcUtil,
    learnUtil,
    DriverMgr,
    PointMgr,
    DepthSubscriber,
    MgrTypeInfo,
    mixinUtils,
    CommandGroup,
    addLearnSupport,
    MgrLearnTableSupport,
    Password,
    passwordServlet
  ) {

  'use strict';

  var applyMixin = mixinUtils.applyMixin,
    hasMixin = mixinUtils.hasMixin,
    logError = log.severe.bind(log),
    N_DISCOVERY_GROUP = 'ndriver:NDiscoveryGroup',
    MIXIN_NAME = 'AUTOMANAGER';

  /*
   * Workbench and UX have differing definitions of subscription depth, so an
   * adjustment is required.
   *
   *  BComponent.java#lease:
   *    If depth is greater than zero then the lease includes descendants
   *    (one is children, two is children and grandchildren, etc).
   *  DepthSubscripber.js#subscribeAll
   *     @param {Number} depth the depth to subscribe - 0 subscribes nothing,
   *     1 subscribes only the given components, 2 subscribes components
   *     and their kids, etc.
   */
  var UX_SUBSCRIPTION_DEPTH_ADJ = 1;

  /*
   * A default subscription depth for the discovery job.
   * 3 caters for BNDiscoveryGroup folder within a BNDiscoveryGroup folder
   */
  var DEFAULT_DISCOVERY_JOB_SUBSCRIPTION_DEPTH = 3;

  /**
   * A mixin to provide Automanager support to ndriver manager views.
   *
   * @alias module:nmodule/ndriver/rc/Automanager
   * @mixin
   * @extends module:nmodule/driver/rc/wb/mgr/DriverMgr
   * @param {module:nmodule/driver/rc/wb/mgr/DriverMgr} target
   * @param {Object} params
   *
   */
  var Automanager = function Automanager(target, params) {

    if (!(target instanceof DriverMgr)) {
      throw new Error('target must be a DriverMgr instance.');
    }

    if (!applyMixin(target, MIXIN_NAME, Automanager.prototype)) {
      return this;
    }

    var superDoInitialize = target.doInitialize,
      superLoad = target.load;

    /**
     * Extend the base Manager's doInitialize() function,
     * adding the jobcomplete handler.
     *
     * @override
     * @param {JQuery} dom
     * @returns {*|Promise}
     */
    target.doInitialize = function (dom, params) {
      var that = this;

      that.on('jobcomplete', function (job) {
        that.$updateLearnTableModelFromJob(job).catch(logError);
      });

      return superDoInitialize.call(that, dom, {});
    };

    /**
     * Extend the base Manager's load() function, looking up/populating automanager properties
     * and adding discovery if required.
     *
     * @override
     * @param {baja.Component} component
     * @returns {Promise}
     */
    target.load = function (component) {
      var that = this, args = arguments;

      return that.setUpManagerInfo(component)
      .then(function (managerInfo) {

        return Promise.all([
          that.$setupForSubscriptionDepth(managerInfo, component),
          that.$setupForDiscovery(managerInfo)
        ]);
      }).then(function () {
        // make sure the types we need are available locally
        return baja.importTypes({
          typeSpecs: that.getTypesToImport()
        });
      }).then(function () {
        return superLoad.apply(that, args);
      });
    };
  };

  /**
   * update this manager's subscriber with the depth configured in BNNetwork.
   *
   * @private
   * @param {Object} managerInfo
   * @param {baja.Component} component
   * @returns {Promise}
   */
  Automanager.prototype.$setupForSubscriptionDepth = function (managerInfo, component) {
    var subscriptionDepth = rpcUtil.getSubscriptionDepthFromResult(managerInfo) + UX_SUBSCRIPTION_DEPTH_ADJ;

    if (!subscriptionDepth || subscriptionDepth === this.getSubscriber().getDepth()) {
      return Promise.resolve();
    }

    this.$subscriber = new DepthSubscriber(subscriptionDepth);

    return this.getSubscriber().subscribe(component);
  };

  /**
   * add discovery setup, if this manager supports discovery.
   *
   * @private
   * @param {Object} managerInfo
   * @returns {Promise}
   */
  Automanager.prototype.$setupForDiscovery = function (managerInfo) {
    var that = this;

    that.$discoveryLeafTypeSpec = rpcUtil.getDiscoveryLeafTypeSpecFromResult(managerInfo);

    if (!that.isDiscoverySupported()) { return Promise.resolve(); }

    // discovery is supported ...
    function filterCommands(cmd) {
      return (hasMixin(cmd, 'MGR_COMMAND')) ? cmd.isShownInActionBar() : true;
    }

    // save the command container as it's already been initialized
    var commandContainerElement = that.jq().find('.commandContainer').detach();

    // replace the standard manager html with the Learn Table template
    that.jq().html(MgrLearnTableSupport.mgrLearnHtml());

    // add the mgr-action-bar class to the standard element, then slot it into the new html
    commandContainerElement.addClass('mgr-action-bar');
    that.jq().find('.commandContainer.mgr-action-bar').replaceWith(commandContainerElement);

    addLearnSupport(that);

    that.makeDiscoveryCommands().forEach(function (discoveryCommand) {
      that.getCommandGroup().add(discoveryCommand);
    });

    /**
     * Extend the base Manager's getExisting() function,
     * delegating the proccessing to the server (and ultimately this ndriver's java
     * implementation of isExisting) via an rpc call.
     *
     * @override
     * @param {baja.Complex} discovery a discovery item
     * @param {baja.comm.Batch} [batch] batch to use for the target `getExisting`
     * @returns {Promise.<baja.Component|undefined>} the existing component that was found to match
     * the given discovery node, or undefined if no such match was found.
     */
    that.getExisting = function (discovery, batch) {
      return rpcUtil.getExisting(discovery, this.value().getNavOrd().toString(), batch);
    };

    return that.initializeJobBar(that.jq())
    .then(function () {
      return Widget.in(that.$getCommandContainerElement().children('.CommandButtonGroup')[1]).load(
        that.getCommandGroup().filter({ include: filterCommands })
      );
    });
  };

  /**
   * An array of type specs that the manager implementation needs
   *
   * @return {Array.<String>} An array of typeSpecs strings
   */
  Automanager.prototype.getTypesToImport = function () {
    var types = this.getManagerTypes();
    if (this.isDiscoverySupported()) {
      types.push(this.getDiscoveryLeafTypeSpec());
    }
    return types;
  };

  /**
   * The typeSpec for the device used by this manager
   *
   * @return {String} the typeSpec for the device
   */
  Automanager.prototype.getDeviceTypeSpec = function () {
    return this.$deviceTypeSpec;
  };

  /**
   * Setup information required by this manager, usually acquired by means of an rpc call to populate
   * values from the ndriver/Automanager interfaces
   *
   * @abstract
   * @param {baja.Component} component - the value being loaded into the manager.
   * @return {Object} key/value pairs of manager info
   */
  Automanager.prototype.setUpManagerInfo = function (component) {
    throw new Error('setUpManagerInfo() function not implemented.');
  };

  /**
   * An array of type specs that this manager implementation needs
   *
   * @abstract
   * @return {Array.<String>}
   */
  Automanager.prototype.getManagerTypes = function () {
    throw new Error('getManagerTypes() function not implemented.');
  };

  /**
   * Provides ndriver automanager specific functionality for adding instances to manager views.
   *
   * @param {Array.<baja.Component>} instances the instances to add
   * @param {Array.<String>} names the desired slot names for the instances.
   * @param {Function} superAddInstances the addInstances function from the super class.
   * @param {module:nmodule/webEditors/rc/wb/mgr/model/MgrModel} mgrModel the base ord for instance being added.
   *
   * @returns {Promise} promise to be resolved when the instances are added.
   */
  Automanager.prototype.addInstancesForNDriver = function (instances, names, superAddInstances, mgrModel) {
    var that = this,
      baseNavOrd = that.value().getNavOrd(),
      passwordChanges = [],
      defaultPassword = Password.DEFAULT;


    var processPasswordSlot = function (slot, parent, baseNavOrd) {
      var pw = parent.get(slot);

      if (!pw.equals(defaultPassword)) {
        // store the details of required password changes for later use

        passwordChanges.push({
          pw: pw.encodeToString(),
          slotName: slot.getName(),
          base: baja.Ord.make(baseNavOrd)
        });
      }

      // instance is unmounted so set will be a sync call.
      parent.set({
        slot: slot,
        value: defaultPassword
      }).catch(logError);
    };

    var processPropertySlots = function (component, baseNavOrd) {
      baseNavOrd = baseNavOrd + '/' + component.getName();
      return component.getSlots().properties().each(function (slot) {
        if (slot.getType().is('baja:Password')) {
          processPasswordSlot(slot, component, baseNavOrd);
        } else if (slot.getType().isComplex() || slot.getType().isStruct()) {
          // if we have a complex or a struct, carry on down the tree
          processPropertySlots(component.get(slot), baseNavOrd);
        }
      });
    };

    // check each instance for slots that can't be saved on unmounted components,
    // specifically ones with passwords
    instances.forEach(function (instance) {
      processPropertySlots(instance, baseNavOrd);
    });

    return superAddInstances.apply(mgrModel, arguments)
      .then(function () {
        // once the parent component is mounted, we can make the password changes
        var batch = new baja.comm.Batch();
        return batch.commit(
          passwordChanges.map(function (passwordChange) {
            return passwordServlet.setPassword(passwordChange.pw, passwordChange.slotName, passwordChange.base, { batch: batch });
          })
        );
      });
  };




  // Discovery ...

  /**
   * The typeSpec for the discovery leaf used by this manager.
   *
   * @return {String}
   */
  Automanager.prototype.getDiscoveryLeafTypeSpec = function () {
    return this.$discoveryLeafTypeSpec;
  };

  /**
   * Whether this manager supports discovery
   *
   * @return {boolean}
   */
  Automanager.prototype.isDiscoverySupported = function () {
    //if getDiscoveryLeafTypeSpec() is falsey it will be taken to mean that discovery is not supported
    return !!this.getDiscoveryLeafTypeSpec();
  };

  /**
   * return the component that has the discovery interface implementations for this manager
   * (the NNetwork for the device manager, or the NPointDeviceExt for the point manager)
   *
   * @return {baja.Component}
   */
  Automanager.prototype.getDiscoveryComponent = function () {
    if (this.isDiscoverySupported()) {
      // only needed if discovery is supported
      throw new Error('getDiscoveryComponent() function not implemented.');
    }
  };

  /**
   * Make the model for the discovery table.
   *
   * @override
   * @function module:webEditors/rc/wb/mgr/MgrLearn#makeLearnModel

   * @returns {Promise.<module:nmodule/webEditors/rc/wb/table/tree/TreeTableModel>}
   */
  Automanager.prototype.makeLearnModel = function () {
    return learnUtil.makeLearnModel(this.getDiscoveryLeafTypeSpec());
  };

  /**
   * Invoke an Action on the station that will submit a discovery job, then
   * set the returned ORD on the manager.
   *
   * @override
   * @function module:webEditors/rc/wb/mgr/MgrLearn#doDiscover
   *
   * @returns {Promise}
   */
  Automanager.prototype.doDiscover = function () {
    var that = this,
      discoveryComp = that.getDiscoveryComponent();

    return Promise.all([
      discoveryComp.submitDiscoveryJob(discoveryComp.getDiscoveryPreferences()),
      that.getDiscoveryJobSubscriptionDepth()
    ])
    .spread(function (ord, depth) {
      ord = baja.Ord.make({
        base: baja.Ord.make('station:'),
        child: ord.relativizeToSession()
      });
      return that.setJob({
        jobOrOrd: ord,
        depth: depth
      });
    });
  };

  /**
   * Return the type(s) suitable for the given discovery item. Some managers may
   * need to inspect the discovery value to return a suitable type or several
   * types.
   *
   * @override
   * @function module:webEditors/rc/wb/mgr/MgrLearn#getTypesForDiscoverySubject
   *
   * @param {baja.Value} discoveredObject
   * @returns {Promise.<module:nmodule/webEditors/rc/wb/mgr/MgrTypeInfo>}
   */
  Automanager.prototype.getTypesForDiscoverySubject = function (discoveredObject) {
    return rpcUtil.getValidDatabaseTypes(discoveredObject)
    .then(function (types) {
      return MgrTypeInfo.make(types);
    });
  };

  /**
   * The subscription depth for the discovery job.
   *
   * @return {Promise.<Number>}
   */
  Automanager.prototype.getDiscoveryJobSubscriptionDepth = function () {
    return Promise.resolve(DEFAULT_DISCOVERY_JOB_SUBSCRIPTION_DEPTH);
  };

  /**
   * Called asynchronously after the job submitted by doDiscover() has
   * finished. This should get the items found in the discovery and
   * update the TreeNodes in the learn table.
   *
   * @private
   * @param {baja.Component} job - the config discovery job.
   * @returns {Promise}
   */
  Automanager.prototype.$updateLearnTableModelFromJob = function (job) {
    var that = this;

    return job.loadSlots()
    .then(function () {
      var discoveries = job.getSlots()
        .is(N_DISCOVERY_GROUP, that.getDiscoveryLeafTypeSpec())
        .toValueArray();

      that.$discoveries = discoveries;
      return that.$updateLearnTable(discoveries);
    });
  };

  /**
   * Function to update the model for the learn table with the discovered
   * items obtained from the job.
   *
   * @private
   * @param {Array.<baja.Component>} discoveries
   * @returns {Promise}
   */
  Automanager.prototype.$updateLearnTable = function (discoveries) {
    var that = this,
      model = that.getLearnModel(),
      currentCount = model.getRows().length;

    return Promise.resolve(currentCount && model.clearRows())
      .then(function () {
        return Promise.all(
          // Create TreeNodes with a value returning the discovered item.
          _.map(discoveries || [], function (discovery) {
            return learnUtil.makeDiscoveryTableNode(
              discovery,
              that.getDiscoveryLeafTypeSpec(),
              (that instanceof PointMgr)
            );
          })
        );
      })
      .then(function (nodes) {
        // Update the model with the discoveries.
        let promises = _.map(nodes, (node) => model.getRootNode().add(node));
        promises.push(model.insertRows(nodes, 0));

        return Promise.all(promises);
      });
  };

  /**
   * Get the values to be set as the proposals on the batch component editor for
   * the rows being added via the AddCommand.
   *
   * @override
   * @function module:webEditors/rc/wb/mgr/MgrLearn#getProposedValuesFromDiscovery
   *
   * @param {baja.Complex} discovery an item obtained from a node in discovery table.
   * @param {*} subject - the subject of the `Row` whose values are to be proposed.
   * @see module:nmodule/webEditors/rc/wb/table/tree/TreeNode
   * @returns {Object|Promise.<Object>} an object literal with the name and initial values to be used for the new component.
   */
  Automanager.prototype.getProposedValuesFromDiscovery = function (discovery, subject) {
    return rpcUtil.getDiscoveryName(discovery)
    .then(function (discoveryName) {
      return {
        name: baja.SlotPath.unescape(discoveryName),
        values: {}
      };
    });
  };

  /**
   * Calls back to the BNDiscoveryLeaf to set up a new instance for this discovery object.
   *
   * @override
   * @function module:webEditors/rc/wb/mgr/MgrLearn#newInstanceFromDiscoverySubject
   *
   * @param {baja.Complex} discovery an instance of a discovery item (e.g. an
   * `ndriver:NDiscoveryLeaf`), dragged from the discovery table and dropped
   * onto the database table or selected when the 'Add' command was invoked.
   *
   * @param {Array.<module:nmodule/webEditors/rc/wb/mgr/MgrTypeInfo>} typeInfos - an
   * array of MgrTypeInfos, created from the type or types returned
   * by the manager's `getTypesForDiscoverySubject()` implementation.
   *
   * @returns {Promise} a Promise of new component instance for the discovered item.
   *
   */
  Automanager.prototype.newInstanceFromDiscoverySubject = function (discovery, typeInfos) {
    var proxyExtTypeSpec = this.getModel().getProxyExtType ? this.getModel().getProxyExtType() : '';
    return rpcUtil.getDefaultDiscoveryInstance(discovery, proxyExtTypeSpec);
  };

  /**
   * Returns a new instance of for the given typeInfo.
   *
   * If a value is supplied for discovery, an rpc call is made to the BNDiscoveryLeaf
   * and returns a new instance of the supplied typeSpec based on the discovery object
   * (Typically called when the Type is changed in the editor).
   *
   * If no discovery value is supplied, the superNewInstance function is called.
   *
   * @param {module:nmodule/webEditors/rc/wb/mgr/MgrTypeInfo} typeInfo
   * @param {Object} [params]
   * @param {baja.Component} [params.discovery] a discovery object
   * @param {Function} newInstance - a newInstance function to call if no discovery object is supplied
   * @param {module:nmodule/webEditors/rc/wb/mgr/model/MgrModel} model
   *
   * @returns {Promise.<baja.Component>} resolves to a new component instance.
   */
  Automanager.prototype.getNewInstance = function (typeInfo, params, newInstance, model) {
    var discovery = params && params.discovery;

    if (discovery) {
      return rpcUtil.getDiscoveryInstance(
        discovery,
        typeInfo.getType().getTypeSpec().toString(),
        this.getModel().getProxyExtType ? this.getModel().getProxyExtType() : ''
      );
    }

    return newInstance.call(model, typeInfo);
  };

  /**
   * Automanager does not implement #isExisting, despite it being required by
   * MgrLearn#getExisting
   * @see {@link module:nmodule/webEditors/rc/wb/mgr/MgrLearn#getExisting}
   * Instead Automanager overrides #getExisting and delegates processing via an
   * rpc call to the java implementation of #isExisting on the DiscoveryLeaf.
   *
   * @param {baja.Complex} discovery
   * @param {baja.Component} component
   *
   * @returns {boolean}  - true if the discovery item and station component match.
   */
  Automanager.prototype.isExisting = function (discovery, component) {
    throw new Error('isExisting() not supported by Automanager');
  };

  // End Discovery



  return (Automanager);
});
