/**
 * @copyright 2020 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module baja/env/mux/BoxFrameCoalescer
 */
define([
  'bajaScript/env/mux/BoxEnvelopeDemux',
  'bajaScript/env/mux/BoxEnvelopeMux',
  'bajaScript/env/mux/muxUtils' ], function (
  BoxEnvelopeDemux,
  BoxEnvelopeMux,
  muxUtils) {

  'use strict';

  var fromBytes = muxUtils.fromBytes;

  /**
   * This class handles the mux/demux of outgoing BoxFrames and their incoming
   * responses.
   *
   * @class
   * @alias module:baja/env/mux/BoxFrameCoalescer
   * @param {object} params
   * @param {module:baja/env/ConnectionManager} params.connectionManager the
   * connection manager to handle incoming box frames
   * @param {number} params.maxEnvelopeSize - the maximum amount of data to
   * buffer up waiting to be sent, in bytes
   * @param {number} params.maxMessageSize - the maximum amount of data that can
   * be sent in one message (e.g. WebSocket message size limit), in bytes
   * @param {number} params.minDelay - the minimum amount of time, in ms, to
   * wait after receiving a BoxFrame for more BoxFrames to be sent, allowing
   * them to coalesce into a single network call
   * @param {number} params.maxDelay - the maximum amount of time, in ms, to
   * wait for more BoxFrames - this prevents a slow, steady trickle of BoxFrames
   * from causing a wait-forever until the envelope buffer is full
   * @param {function} params.sendToServer - a function that receives a string
   * and sends up to the BOX servlet
   */
  var BoxFrameCoalescer = function BoxFrameCoalescer(params) {
    var that = this;

    that.$demux = new BoxEnvelopeDemux({
      onComplete: function (env) { that.completeEnvelope(env); }
    });

    that.$mux = new BoxEnvelopeMux({
      maxEnvelopeSize: params.maxEnvelopeSize,
      maxMessageSize: params.maxMessageSize,
      minDelay: params.minDelay,
      maxDelay: params.maxDelay,
      sendToServer: params.sendToServer
    });

    that.$connectionManager = params.connectionManager;

    /**
     * Map of envelope id -> FrameData[] - BoxFrames that are
     * waiting for their callbacks to resolve.
     * @type {Object<number, Array<module:baja/env/Connection~FrameData>>}
     */
    that.$pendingFrames = {};
  };

  /**
   * @param {module:baja/env/Connection~FrameData} frameData an outgoing
   * BoxFrame
   */
  BoxFrameCoalescer.prototype.sendFrameData = function (frameData) {
    // buffer it in mux
    // sock it away by envelope id/frame number
    var frameBody = frameData.frame.$body;
    var envelopeId = this.$mux.send({ m: frameBody.m, n: frameBody.n });

    var pendingFrames = this.$pendingFrames;
    var frameDatas = pendingFrames[envelopeId] || (pendingFrames[envelopeId] = []);
    frameDatas.push(frameData);
  };

  /**
   * @param {module:baja/env/mux/BoxEnvelope~BoxFragment} fragment
   */
  BoxFrameCoalescer.prototype.receiveFragment = function (fragment) {
    this.$demux.receiveFragment(fragment);
  };

  /**
   * @param {module:baja/env/mux/BoxEnvelope} env a completed incoming
   * BoxEnvelope (solicited or unsolicited)
   */
  BoxFrameCoalescer.prototype.completeEnvelope = function (env) {
    var envelopeId = env.getEnvelopeId();
    var unsolicited = env.isUnsolicited();
    var pendingFrames = this.$pendingFrames;

    if (!unsolicited && !pendingFrames[envelopeId]) {
      // received reply to envelope id not requested - likely due to connection sharing
      return;
    }

    var boxFrames = JSON.parse(fromBytes(env.getPayload()));
    var connectionManager = this.$connectionManager;

    boxFrames.forEach(function (frame) { connectionManager.receiveFrame(frame); });
    if (!unsolicited) { delete pendingFrames[envelopeId]; }
  };

  return BoxFrameCoalescer;
});
