/**
 * @copyright 2020 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/* jshint browser: true *//* eslint-env browser */

/**
 * API Status: **Private**
 * @module baja/env/mux/BoxEnvelopeMux
 */
define([
  'bajaScript/sys',
  'bajaScript/env/mux/BoxEnvelope' ], function (
  baja,
  BoxEnvelope) {

  'use strict';

  /**
   * This class does the work of debouncing individual chunks of data as they
   * are sent from the client, bundling them up into envelopes and sending the
   * individual fragments across the wire.
   *
   * @class
   * @alias module:baja/comm/BoxEnvelopeMux
   * @param params
   * @param {Function} params.sendToServer - a function that receives a string
   * and sends it up to the BOX servlet
   * @param {number} params.maxEnvelopeSize - maximum amount of data to have
   * pending at any one time
   * @param {number} params.maxMessageSize - maximum amount of data to send in
   * a single message (e.g. WebSocket size limit)
   * @param {number} params.minDelay - "debounce" time - minimum amount of time
   * to delay a frame from sending so that more frames can be coalesced into a
   * single message
   * @param {number} params.maxDelay - the maximum amount of time to hold onto
   * pending data before sending. This would prevent a slow, steady trickle of
   * BoxFrames from just queueing forever.
   */
  var BoxEnvelopeMux = function BoxEnvelopeMux(params) {
    this.$sendToServer = params.sendToServer;
    this.$maxEnvelopeSize = params.maxEnvelopeSize;
    this.$maxMessageSize = params.maxMessageSize;
    this.$minDelay = params.minDelay;
    this.$maxDelay = params.maxDelay;

    /**
     * BoxFrames that are sitting, waiting for their timeout to elapse so they
     * can be sent up to the server.
     * @type {module:baja/env/mux/BoxEnvelope}
     */
    this.$bufferEnvelope = null;
  };

  /**
   * @param {string} data - data to be JSON-stringified and sent
   * @returns {number} the id of the BoxEnvelope in which the data will be sent
   */
  BoxEnvelopeMux.prototype.send = function (data) {
    var that = this;
    var envData = BoxEnvelope.toBytes(JSON.stringify(data));

    if (envData.length > this.$maxEnvelopeSize) {
      throw new Error('single message overflowed envelope size (see box.mux.maxEnvelopeSize system property)');
    }

    if (!that.$canBuffer(envData)) {
      that.$flush();
    }

    that.$buffer(envData);
    that.$clearTimeout(that.$minTicket);
    that.$minTicket = that.$flushAfter(that.$minDelay);
    that.$maxTicket = that.$flushAfter(that.$maxDelay);
    return this.$getBufferEnvelope().getEnvelopeId();
  };

  /**
   * @private
   * @returns {module:baja/env/mux/BoxEnvelope} the envelope currently used to
   * buffer up outgoing data
   */
  BoxEnvelopeMux.prototype.$getBufferEnvelope = function () {
    return this.$bufferEnvelope || (this.$bufferEnvelope = this.$startFreshEnvelope());
  };

  /**
   * @private
   * @param {number} delay in ms
   * @returns {number} timeout ticket
   */
  BoxEnvelopeMux.prototype.$flushAfter = function (delay) {
    var that = this;
    return that.$setTimeout(function () { that.$flush(); }, delay);
  };

  /**
   * @private
   * @returns {module:baja/env/mux/BoxEnvelope}
   */
  BoxEnvelopeMux.prototype.$startFreshEnvelope = function () {
    return BoxEnvelope.make({
      sessionId: baja.comm.getServerSessionId(),
      envelopeId: baja.$mux.nextEnvelopeId(),
      maxMessageSize: this.$maxMessageSize,
      maxEnvelopeSize: this.$maxEnvelopeSize
    });
  };

  /**
   * @private
   * @param {Uint8Array} payload
   * @returns {boolean} if the payload can be buffered without overflowing the
   * buffer envelope
   */
  BoxEnvelopeMux.prototype.$canBuffer = function (payload) {
    // two bytes for array brackets (if empty) or comma and close bracket (if not empty)
    return this.$getBufferEnvelope().willFit(payload.length + 2);
  };

  /**
   * @private
   * @param {Uint8Array|string} data a string of JSON
   */
  BoxEnvelopeMux.prototype.$buffer = function (data) {
    var env = this.$getBufferEnvelope();
    env.append(this.$bufferEmpty() ? '[' : ',');
    env.append(data);
  };

  /**
   * @private
   * @returns {boolean}
   */
  BoxEnvelopeMux.prototype.$bufferEmpty = function () {
    return this.$getBufferEnvelope().isEmpty();
  };

  /**
   * @private
   */
  BoxEnvelopeMux.prototype.$flush = function () {
    var that = this;
    var env = that.$getBufferEnvelope();
    var sendToServer = that.$sendToServer;

    that.$clearTimeout(that.$minTicket);
    that.$clearTimeout(that.$maxTicket);

    if (!this.$bufferEmpty()) {
      env.append(']');
      env.toBoxFragments().forEach(sendToServer);
      that.$bufferEnvelope = that.$startFreshEnvelope();
    }
  };

  // eslint-disable-next-line no-constant-condition
  var global = (typeof jasmine !== 'undefined' &&
    jasmine.Clock &&
    jasmine.Clock.real) ? jasmine.Clock.real : window;
  var setTimeout = global.setTimeout;
  var clearTimeout = global.clearTimeout;

  /*
   * we grab a permanent reference to the "real" setTimeout/clearTimeout.
   * otherwise, when a test uses jasmine.Clock.useMock(), and muxing is enabled,
   * it basically stops BajaScript.
   */
  BoxEnvelopeMux.prototype.$setTimeout = function (func, delay) {
    // noinspection DynamicallyGeneratedCodeJS
    return setTimeout(func, delay);
  };

  BoxEnvelopeMux.prototype.$clearTimeout = function (ticket) {
    return clearTimeout(ticket);
  };

  return BoxEnvelopeMux;
});
