/**
 * @copyright 2020 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/* jshint browser: true *//* eslint-env browser */

/**
 * API Status: **Private**
 * @module baja/env/mux/BoxEnvelopeDemux
 */
define([
  'bajaScript/env/mux/BoxEnvelope' ], function (
  BoxEnvelope) {

  'use strict';

  /**
   * This class waits for fragments to be received from the server, assembles
   * them into full envelopes, and notifies on completion.
   *
   * Because it keeps track of open envelopes by envelope ID, it sorts them into
   * two piles: solicited, which are replies to envelopes sent up by the client,
   * and unsolicited, which are sent directly from the server. This is because
   * the two envelope ID counters are not in sync with each other and may
   * overlap.
   *
   * @class
   * @alias module:baja/env/mux/BoxEnvelopeDemux
   * @param {object} params
   * @param {function} params.onComplete callback to receive envelopes as they
   * are completed
   */
  var BoxEnvelopeDemux = function BoxEnvelopeDemux(params) {
    var that = this;
    that.$onComplete = params.onComplete || function () {};

    /**
     * Keeps track of which envelopes are currently open and waiting to be
     * completed by receiving all envelope fragments.
     * @type {module:baja/env/mux/BoxEnvelopeDemux~EnvelopeStaging}
     */
    that.$pendingEnvelopes = { s: {}, u: {} };
  };

  /**
   * @param {number} envelopeId
   * @param {boolean} unsolicited true if this is an unsolicited envelope sent
   * down from the server
   * @returns {module:baja/env/mux/BoxEnvelope} the opened envelope (or
   * existing if already open)
   */
  BoxEnvelopeDemux.prototype.openEnvelope = function (envelopeId, unsolicited) {
    var that = this;
    var pendingEnvelopes = that.$pendingEnvelopes;
    var envelope = getFromStaging(pendingEnvelopes, envelopeId, unsolicited);

    if (envelope) { return envelope; }

    envelope = putInStaging(pendingEnvelopes, envelopeId, unsolicited,
      new BoxEnvelope({ envelopeId: envelopeId, unsolicited: unsolicited }));

    return envelope;
  };

  /**
   * @param {number} envelopeId
   * @param {boolean} unsolicited
   * @returns {module:baja/env/mux/BoxEnvelope}
   */
  BoxEnvelopeDemux.prototype.getPendingEnvelope = function (envelopeId, unsolicited) {
    return getFromStaging(this.$pendingEnvelopes, envelopeId, unsolicited);
  };

  /**
   * @param {module:baja/env/mux/BoxEnvelope~BoxFragment} fragment
   */
  BoxEnvelopeDemux.prototype.receiveFragment = function (fragment) {
    var envelopeId = fragment.envelopeId;
    var fragmentCount = fragment.fragmentCount;
    var fragmentIndex = fragment.fragmentIndex;
    var unsolicited = fragment.unsolicited;
    var payload = fragment.payload;

    var envelope = this.getPendingEnvelope(envelopeId, unsolicited) ||
      this.openEnvelope(envelopeId, unsolicited);

    envelope.$fragmentCount = fragmentCount;
    envelope.receiveFragment(payload, fragmentIndex);

    if (envelope.isComplete()) {
      this.$remove(envelopeId, unsolicited);
      this.$onComplete(envelope);
    }
  };

  /**
   * @private
   * @param {number} envelopeId
   * @param {boolean} unsolicited
   */
  BoxEnvelopeDemux.prototype.$remove = function (envelopeId, unsolicited) {
    deleteFromStaging(this.$pendingEnvelopes, envelopeId, unsolicited);
  };

  function getFromStaging(staging, id, unsolicited) {
    return getStaging(staging, unsolicited)[id];
  }

  function putInStaging(staging, id, unsolicited, envelope) {
    return (getStaging(staging, unsolicited)[id] = envelope);
  }

  function deleteFromStaging(staging, id, unsolicited) {
    delete getStaging(staging, unsolicited)[id];
  }

  function getStaging(staging, unsolicited) {
    return staging[unsolicited ? 'u' : 's'];
  }

  /**
   * @private
   * @typedef module:baja/env/mux/BoxEnvelopeDemux~EnvelopeStaging
   * @property {Object<number, module:baja/env/mux/BoxEnvelope>} s a mapping
   * of envelope id -> BoxEnvelope for solicited envelopes
   * @property {Object<number, module:baja/env/mux/BoxEnvelope>} u a mapping
   * of envelope id -> BoxEnvelope for unsolicited envelopes
   */

  return BoxEnvelopeDemux;
});
