/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/*jshint browser: true *//* eslint-env browser */

/**
 * WebSocket Connection.
 *
 * API Status: **Private**
 * @module baja/env/WebSocketConnection
 */
define([
  "bajaScript/comm",
  "bajaScript/baja/comm/Callback",
  "bajaScript/env/Connection" ], function (
  baja,
  Callback,
  Connection) {

  "use strict";

  // for artificial websocket latency performance testing
  // var minDelay = 0;
  // var webSocketLatency = 0;

  /**
   * A connection that uses a WebSocket as a connection.
   *
   * @class
   * @extends module:baja/env/Connection
   * @alias module:baja/env/WebSocketConnection
   * @param {module:baja/env/Connection~ConnectionEvents} handlers handlers
   * for connection events.
   * @param {module:baja/env/ConnectionManager} connectionManager the connection
   * manager responsible for processing BoxFrames via incoming websocket
   * messages
   */
  var WebSocketConnection = function (handlers, connectionManager) {
    baja.callSuper(WebSocketConnection, this, arguments);
  };

  baja.subclass(WebSocketConnection, Connection);

  /**
   * Make a new server session. This must be done using the default HTTP
   * mechanism before a web socket can be opened.
   * @return {Promise} promise to be resolved after the server session is
   * created
   */
  WebSocketConnection.prototype.doAuthenticate = function () {
    var cb = new Callback();
    baja.comm.makeServerSession(cb);
    return cb.promise();
  };

  /**
   * Opens a new web socket to the station.
   */
  WebSocketConnection.prototype.start = function () {
    var that = this,
        handlers = that.$handlers,
        connection = that.$internalConnection = that.$createSocket();

    for (var name in handlers) {
      if (handlers.hasOwnProperty(name)) {
        connection.addEventListener(name, handlers[name], false);
      }
    }
  };

  /**
   * @private
   * @returns {WebSocket} web socket pointed at the BOX WebSocket acceptor
   */
  WebSocketConnection.prototype.$createSocket = function () {
    var isSecure = /^https/i.test(window.location.protocol),
      protocol = isSecure ? 'wss' : 'ws',
      uri = protocol + '://' + window.location.host + '/wsbox';

    // TODO: We should list supported protocols in the WebSocket constructor?
    // https://developer.mozilla.org/en-US/docs/WebSockets/Writing_WebSocket_client_applications

    try {
      return new WebSocket(uri);
    } catch (e) {
      baja.error('WebSockets are available in this browser, but could not ' +
        'construct a WebSocket connection to ' + uri);
      throw e;
    }
  };

  return WebSocketConnection;
});
