/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * @module baja/tag/TagSet
 */
define(["bajaScript/sys",
        "bajaScript/baja/tag/Id"], function (
        baja,
        Id) {
  
  "use strict";
     
  /**
   * Tags is used to store a collection of {@link module:baja/tag/Tag} 
   * objects. 
   *
   * This implementation is a simple collection.
   *
   * @class
   * @alias module:baja/tag/TagSet
   *
   * @param {Array<module:baja/tag/Tag>} tags
   * An array of tags.
   */  
  var TagSet = function TagSet(tags) {
    this.$tags = tags || [];
  };

  /**
   * @returns {Boolean} Returns true if there are no Tag objects.
   */
  TagSet.prototype.isEmpty = function() {
    return this.$tags.length === 0;
  };

  /**
   * Return the value for the specified tag.
   * 
   * @param  {String|module:baja/tag/Id} id The Id
   * used for the search. This can be an Id or a qname for an Id.
   * @returns The value for the tag or null if the tag doesn't exist.
   */
  TagSet.prototype.get = function(id) {
    var i,
        tags = this.$tags;

    id = typeof id === "string" ? new Id(id) : id;

    for (i = 0; i < tags.length; ++i) {
      if (tags[i].getId().equals(id)) {
        return tags[i].getValue();
      }
    }

    return null;
  };

  /**
   * Returns true a Tag with the specified Id (or qname)
   * is found.
   * 
   * @param  {String|module:baja/tag/Id} id The Id
   * used for the search. This can be an Id or a qname for an Id.
   * @returns {Boolean} Returns true if found.
   */
  TagSet.prototype.contains = function(id) {
    return this.get(id) !== null;
  };

  /**
   * Returns a copy of the contained Tags array.
   * 
   * @returns {Array<module:baja/tag/Tag>} An array of Tag objects.
   */
  TagSet.prototype.getAll = function() {
    return Array.prototype.slice.call(this.$tags);
  };

  return TagSet;
});