/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * @module baja/tag/Id
 */
define(["bajaScript/baja/ord/SlotPath"], function (SlotPath) {
  
  "use strict";
     
  /**
   * An id is an immutable dictionary/name code pair. An id uniquely
   * identifies (Tag)s and (Relation)s.
   *
   * @class
   * @alias module:baja/tag/Id
   *
   * @param {String} dictionary The dictionary to be used. Or if only argument
   * is being specified, this should be the qname for the Id.
   * @param {String} [name] The name of Id. If the first argument is the qname,
   * this parameter is optional.
   */  
  var Id = function Id(dictionary, name) {
    var that = this,
        qname,
        res;

    // Is qname?
    if (arguments.length === 1) {
      res = /^(?:([^:]+):)?([^:]+)$/.exec(dictionary);
      if (!res) {
        throw new Error("Invalid qname for Id: " + dictionary);
      }

      qname = dictionary;
      dictionary = res[1] || "";
      name = res[2];
    }

    if (!/^[^:]*$/.test(dictionary)) {
      throw new Error("Invalid dictionary for Id: " + dictionary);
    }

    if (!/^[^:]+$/.test(name)) {
      throw new Error("Invalid name for Id: " + name);
    }

    that.$dict = dictionary;
    that.$name = name;
    that.$qname = qname || (dictionary ? dictionary + ":" + name : name);
  };

  /**
   * Return the Id's dictionary.
   * 
   * @returns {String} The dictionary name or a blank string if 
   * there's no dictionary.
   */
  Id.prototype.getDictionary = function () {
    return this.$dict;
  };

  /**
   * Return true if a dictionary is available.
   * 
   * @returns {Boolean} Returns true if a dictionary 
   * is available.
   */
  Id.prototype.hasDictionary = function () {
    return !!this.$dict;
  };

  /**
   * Return the name of the Id.
   * 
   * @returns {String} Returns the name.
   */
  Id.prototype.getName = function () {
    return this.$name;
  };

  /**
   * Return the qname of the Id.
   * 
   * @returns {String} Returns the qname.
   */
  Id.prototype.getQName = function () {
    return this.$qname;
  };

  /**
   * Return the string representation of an Id.
   * 
   * @returns {String} Returns the string representation of an Id.
   */
  Id.prototype.toString = function () {
    return this.getQName();
  };

  /**
   * Check for one Id equaling another.
   * 
   * @param arg The argument to be tested.
   * @returns {Boolean} true if the same.
   */
  Id.prototype.equals = function (arg) {
    return !!(arg && this.$qname === arg.$qname);
  };

  /**
   * Return a facet key for the specified id.
   * 
   * @param  {module:baja/tag/Id|String} id The id
   * or qname if the id to turn into a facet key.
   * @returns {String} The encoded key.
   */
  Id.idToFacetKey = function (id) {
    return SlotPath.escape(id.toString());
  };

  /**
   * Return an Id from the facet key.
   *
   * @param {String} key The facet key to create an
   * Id from.
   * @returns {module:baja/tag/Id} An
   * Id created from the facet key.
   */
  Id.facetKeyToId = function (key) {
    return new Id(SlotPath.unescape(key));
  };

  return Id;
});