/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * Defines {@link baja.OrdScheme}.
 * @module baja/ord/OrdScheme
 */
define([
  "bajaScript/sys",
  "bajaScript/baja/obj/Singleton",
  "bajaScript/baja/ord/OrdQuery" ], function (
  baja,
  Singleton,
  OrdQuery) {
  
  "use strict";
  
  var subclass = baja.subclass,
      callSuper = baja.callSuper;
  
  /**
   * ORD Scheme.
   * 
   * An ORD is made up of a series of ORD Queries separated by the '|' character.
   * Each ORD Query has an ORD Scheme name (e.g. 'slot') and a body (e.g. 
   * '/Drivers/ModbusNetwork').
   *
   * @class
   * @alias baja.OrdScheme
   * @extends baja.Singleton
   */     
  var OrdScheme = function OrdScheme() {
    callSuper(OrdScheme, this, arguments);
  };
  
  subclass(OrdScheme, Singleton);
  
  /**
   * All extended ORD Schemes must implement this method so an ORD can be resolved!
   *
   * @abstract
   * @param {module:baja/ord/OrdTarget} target  the current ORD Target
   * @param {baja.OrdQuery} query  the ORD Query used in resolving the ORD
   * @param {module:baja/ord/OrdQueryListCursor} cursor  the ORD Query List 
   * cursor used for helping to asynchronously resolve the ORD
   * @param {Object} options  options used for resolving an ORD
   */
  OrdScheme.prototype.resolve = function (target, query, cursor, options) {
    throw new Error("ORD Scheme must implement resolve: " + this.getType());
  };

  /**
   * @returns {boolean} true if this scheme has logic implemented to resolve
   * client-side. By default, returns true - typically the whole reason to
   * implement a `baja.OrdScheme` is to add client-side resolution logic. If
   * this function returns false, it indicates that only the server knows how to
   * resolve an ORD containing this scheme.
   * @since Niagara 4.10
   * @see baja.OrdQueryList#isClientResolvable
   */
  OrdScheme.prototype.isClientResolvable = function () {
    return true;
  };
      
  /**
   * Return the ORD Scheme for the given scheme name.
   *
   * @returns {baja.OrdScheme}
   */
  OrdScheme.lookup = function (schemeName) {
    var type = baja.registry.getOrdScheme(schemeName);
    return type && type.hasConstructor() ? type.getInstance() : baja.UnknownScheme.DEFAULT;
  };
  
  /**
   * Return an ORD Query for the scheme.
   *
   * @returns {baja.OrdQuery}
   */
  OrdScheme.prototype.parse = function (schemeName, body) {
    return new OrdQuery({
      scheme: this,
      schemeName: schemeName,
      body: body
    });
  };
  
  return OrdScheme;
});
