/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * Defines `OrdQueryListCursor` (not exposed on `baja` namespace).
 * @module baja/ord/OrdQueryListCursor
 */
define(["bajaScript/sys"], 
    function (baja) {
  
  "use strict";
  
  var subclass = baja.subclass,
      callSuper = baja.callSuper;
  
  /**
   * Cursor for an ORD Query List.
   * 
   * @class
   * @alias module:baja/ord/OrdQueryListCursor
   * @extends baja.SyncCursor
   * @inner
   * @public
   * @param {baja.OrdQueryList} list
   */
  var OrdQueryListCursor = function OrdQueryListCursor(list) {
    callSuper(OrdQueryListCursor, this, arguments);
    this.$list = list;
    this.$index = -1;
  };
  
  subclass(OrdQueryListCursor, baja.SyncCursor);
  
  /**
   * Return true if there's another query in the Cursor.
   *
   * @returns {Boolean}
   */
  OrdQueryListCursor.prototype.hasNext = function () {
    return this.$index + 1 < this.$list.$queries.length;
  };

  /**
   * Advance the Cursor to the next query.
   *
   * @returns {Boolean} true if there's another query in the Cursor.
   */
  OrdQueryListCursor.prototype.next = function () {
    if (!this.hasNext()) {
      return false;
    }

    this.$index++;
    return true;
  };
  
  /**
   * Return the current query from the Cursor.
   *
   * @returns {baja.OrdQuery}
   */
  OrdQueryListCursor.prototype.get = function () {
    if (this.$index === -1) {
      throw new Error("Illegal cursor index");
    }
    return this.$list.$queries[this.$index];
  };
  
  /**
   * Iterate through the Cursor and call a function on every item.
   * 
   * @param {Function} func function called on every iteration with the
   * {@link baja.OrdQuery} being used as an argument.
   */
  OrdQueryListCursor.prototype.each = function (func) {
    var result;
    while (this.next()) {
      result = func(this.get(), this.$index);
      if (result) {
        return result;
      }
    }
  };
  
  /**
   * Return the current index for the Cursor.
   *
   * @returns {Number}
   */
  OrdQueryListCursor.prototype.getIndex = function () {
    return this.$index;
  };
  
  /**
   * Return the current ORD String at the current index.
   *
   * @returns {String}
   */
  OrdQueryListCursor.prototype.getOrd = function () {
    return this.$list.toString(this.$index + 1);
  };
  
  /**
   * Resolve the next ORD Query.
   *
   * @param {module:baja/ord/OrdTarget} target
   * @param {Object} options
   */
  OrdQueryListCursor.prototype.resolveNext = function (target, options) {
    try {
      // Resolve the next part of the ORD scheme
      if (this.next()) {
        var query = this.get();
        query.getScheme().resolve(target, query, this, options);
      }
      // If we've finished iterating through the ORDs then
      // call the original ok callback
      else {
        options.callback.ok(target);
      }
    }
    catch (err) {
      options.callback.fail(err);
    }
  };
  
  return OrdQueryListCursor;
});