/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Jeremy Narron
 */

/**
 * Defines {@link baja.OrdList}
 * @module baja/ord/OrdList
 */
define(['bajaScript/nav',
        'bajaScript/baja/obj/objUtil',
        'bajaScript/baja/obj/Simple',
        'bajaScript/baja/ord/Ord'], function (
        baja,
        objUtil,
        Simple,
        Ord) {

  'use strict';

  var cacheDecode = objUtil.cacheDecode,
      cacheEncode = objUtil.cacheEncode;

  /**
   * A list of `Ord`s.
   *
   * This constructor shouldn't be invoked directly. Please use the `make()`
   * function to create an instance of an OrdList.
   *
   * @see baja.OrdList.make
   *
   * @class
   * @alias baja.OrdList
   * @extends baja.Simple
   */
  var OrdList = function OrdList() {
    Simple.apply(this, arguments);
  };
  OrdList.prototype = Object.create(Simple.prototype);
  OrdList.prototype.constructor = OrdList;

  /**
   * Make a new instance of an OrdList.
   *
   * The argument must be an `Array` that only contains `String`s and `Ord`s.
   *
   * @param {Array.<String|baja.Ord>} ords
   * @returns {baja.OrdList}
   */
  OrdList.make = function (ords) {
    if (!Array.isArray(ords)) {
      throw new Error('array required');
    }

    var ordList = new OrdList();
    ordList.$ords = ords.map(Ord.make);
    return ordList;
  };

  /**
   * Make a new instance of an OrdList.
   *
   * @see baja.OrdList.make
   *
   * @param {Array.<String|baja.Ord>} ords
   * @returns {baja.OrdList}
   */
  OrdList.prototype.make = function (ords) {
    return OrdList.make.apply(OrdList, arguments);
  };

  /**
   * Decode an OrdList from a `String`.
   *
   * @param {String} str the OrdList `String`
   * @returns {baja.OrdList} the decoded OrdList
   */
  OrdList.prototype.decodeFromString = cacheDecode(function (str) {
    if (!str) {
      return OrdList.DEFAULT;
    }
    return OrdList.make(str.split('\n'));
  });

  /**
   * Encode an OrdList to a `String`.
   *
   * @returns {String} the OrdList encoded to a `String`
   */
  OrdList.prototype.encodeToString = cacheEncode(function () {
    return this.$ords.join('\n');
  });

  /**
   * Get the full list of `Ord`s.
   *
   * @returns {Array}
   */
  OrdList.prototype.getOrds = function () {
    return this.$ords.slice();
  };

  /**
   * Get the number of `Ord`s in the OrdList.
   *
   * @returns {Number}
   */
  OrdList.prototype.size = function () {
    return this.$ords.length;
  };

  /**
   * Default OrdList instance.
   *
   * @type {baja.OrdList}
   */
  OrdList.DEFAULT = OrdList.make([]);

  return OrdList;
});
