/**
 * @copyright 2020 Tridium, Inc. All Rights Reserved.
 * @author Vikram Nagulan
 */

/**
 * Defines {@link baja.NSpaceScheme}.
 * @module baja/ord/NSpaceScheme
 */
define([
  "bajaScript/sys",
  "bajaScript/baja/ord/OrdQuery",
  "bajaScript/baja/ord/OrdScheme",
  "bajaScript/baja/ord/OrdTarget"
], function (
  baja,
  OrdQuery,
  OrdScheme,
  OrdTarget
) {
  'use strict';

  var subclass = baja.subclass,
    callSuper = baja.callSuper;

  /**
   * NSpace ORD Scheme.
   * 
   * This scheme resolves to Component Space or Virtual Component Space.
   *
   * @class
   * @alias baja.NSpaceScheme
   * @extends baja.OrdScheme
   * @private
   */
  var NSpaceScheme = function NSpaceScheme() {
    callSuper(NSpaceScheme, this, arguments);
  };

  subclass(NSpaceScheme, OrdScheme);

  /**
   * Default NSpace ORD Scheme instance.
   * @private
   * @type {baja.NSpaceScheme}
   */
  NSpaceScheme.DEFAULT = new NSpaceScheme();

  NSpaceScheme.prototype.resolve = function (target, query, cursor, options) {
    var body = query.getBody();
    var newTarget = new OrdTarget(target);
    newTarget.object = target.object;
    if (!body) {
      newTarget.object = baja.nav.localhost.station;
      return cursor.resolveNext(newTarget, options);
    }

    return baja.rpc({
      typeSpec: "niagaraDriver:NiagaraNetwork",
      method: "getStationSpaceOrdByName",
      args: [ body ]
    })
      .then(function (stationSpaceOrd) {
        return baja.Ord.make(stationSpaceOrd).resolve()
          .then(function (target) {
            newTarget.object = target.object;
            return cursor.resolveNext(newTarget, options);
          });
      })
      .catch(function (err) {
        options.callback.fail(err);
      });
  };

  /**
   * Return an ORD Query for the scheme.
   *
   * @returns {baja.OrdQuery}
   */
  NSpaceScheme.prototype.parse = function (schemeName, body) {
    return new OrdQuery({
      scheme: this,
      schemeName: schemeName,
      body: body,
      isHost: false,
      isSession: false
    });
  };

  return NSpaceScheme;
});
