/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * Defines {@link baja.HttpScheme}.
 * @module baja/ord/HttpScheme
 */
define(["bajaScript/sys",
        "bajaScript/baja/ord/OrdQuery",
        "bajaScript/baja/ord/OrdScheme",
        "bajaScript/baja/ord/OrdTarget",
        "bajaScript/baja/ord/ordUtil"], 
    function (baja, OrdQuery, OrdScheme, OrdTarget, ordUtil) {
  
  "use strict";
  
  var subclass = baja.subclass,
      callSuper = baja.callSuper,
      trimToStart = ordUtil.trimToStart;
  
  /**
   * HTTP ORD Scheme. 
   *
   * @class
   * @alias baja.HttpScheme
   * @extends baja.OrdScheme
   * @private
   */    
  var HttpScheme = function HttpScheme() {
    callSuper(HttpScheme, this, arguments);
  };
  
  subclass(HttpScheme, OrdScheme);
  
  /**
   * Default HTTP ORD Scheme instance.
   * @private
   * @type {baja.HttpScheme}
   */
  HttpScheme.DEFAULT = new HttpScheme();
  
  /**
   * Called when an ORD is resolved.
   *
   * @private
   *
   * @see baja.OrdScheme#resolve
   *
   * @param {module:baja/ord/OrdTarget} target  the current ORD Target.
   * @param {baja.OrdQuery} query  the ORD Query used in resolving the ORD.
   * @param {module:baja/ord/OrdQueryListCursor} cursor  the ORD Query List 
   * cursor used for helping to asynchronously resolve the ORD.
   * @param {Object} options  options used for resolving an ORD.
   */
  HttpScheme.prototype.resolve = function (target, query, cursor, options) {
    cursor.resolveNext(new OrdTarget(target), options);
  };
      
  /**
   * Return an ORD Query for the scheme.
   *
   * @returns {baja.OrdQuery}
   */
  HttpScheme.prototype.parse = function (schemeName, body) {
    return new OrdQuery({
      scheme: this,
      schemeName: schemeName,
      body: body,
      normalize: trimToStart
    });
  };
  
  return HttpScheme;
});