/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * Defines {@link baja.FoxScheme}.
 * @module baja/ord/FoxScheme
 */
define(["bajaScript/sys",
        "bajaScript/baja/ord/OrdQuery",
        "bajaScript/baja/ord/OrdScheme",
        "bajaScript/baja/ord/OrdTarget"], 
    function (baja, OrdQuery, OrdScheme, OrdTarget) {
  
  "use strict";
  
  var subclass = baja.subclass,
      callSuper = baja.callSuper;
  
  /**
   * Fox ORD Scheme.
   *
   * @class
   * @alias baja.FoxScheme
   * @extends baja.OrdScheme
   * @private
   */    
  var FoxScheme = function FoxScheme() {
    callSuper(FoxScheme, this, arguments);
  };
  
  subclass(FoxScheme, OrdScheme);

  /**
   * Default Fox ORD Scheme instance.
   * @private
   * @type {baja.FoxScheme}
   */
  FoxScheme.DEFAULT = new FoxScheme();
  
  /**
   * Called when an ORD is resolved.
   *
   * @private
   *
   * @see baja.OrdScheme#resolve
   *
   * @param {module:baja/ord/OrdTarget} target  the current ORD Target.
   * @param {baja.OrdQuery} query  the ORD Query used in resolving the ORD.
   * @param {module:baja/ord/OrdQueryListCursor} cursor the ORD Query List 
   * cursor used for helping to asynchronously resolve the ORD.
   * @param {Object} options  options used for resolving an ORD.
   */
  FoxScheme.prototype.resolve = function (target, query, cursor, options) {
    var newTarget = new OrdTarget(target);
    newTarget.object = target.object;
    cursor.resolveNext(newTarget, options);
  };
  
  /**
   * Return an ORD Query for the scheme.
   *
   * @returns {baja.OrdQuery}
   */
  FoxScheme.prototype.parse = function (schemeName, body) {
    return new OrdQuery({
      scheme: this,
      schemeName: schemeName,
      body: body,
      isSession: true,
      normalize: function (list, index) {
        // Shift to host
        var i, q, modified = false;
        for (i = index - 1; i >= 0; --i) {
          q = list.get(i);
          if (!q.isHost()) {
            list.remove(i);
            modified = true;
          }
        }
        return modified;
      }
    });
  };
  
  return FoxScheme;
});