/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * Defines {@link baja.DefaultSimple}.
 * @module baja/obj/DefaultSimple
 */
define(["bajaScript/sys",
        "bajaScript/baja/obj/Simple"], 
        function (baja, Simple) {
  
  'use strict';
  
  var subclass = baja.subclass,
      callSuper = baja.callSuper,
      strictArg = baja.strictArg;
  
  /**
   * Most of the core `Simple`s in BajaScript are represented (e.g. 
   * `baja:String`, `baja:Double` etc). However, there will always be some 
   * `Simple`s that BajaScript won't have support for. If a dedicated JS 
   * constructor for a `Simple` can't be found, it will default back to an 
   * instance of a `DefaultSimple`.
   * 
   * A `DefaultSimple` holds the decoded `String` representation of a `Simple`.
   * 
   * When creating a `Simple`, always use the `make()` method instead of 
   * creating a new Object.
   *
   * @class
   * @alias baja.DefaultSimple
   * @extends baja.Simple
   */
  var DefaultSimple = function DefaultSimple(val) {
    callSuper(DefaultSimple, this, arguments);    
    this.$val = strictArg(val || "", String);
  };
  
  subclass(DefaultSimple, Simple);
  
  /**
   * Default `DefaultSimple` instance.
   * @type {baja.DefaultSimple}
   */
  DefaultSimple.DEFAULT = new DefaultSimple();
    
  /**
   * Make a `DefaultSimple`.
   *
   * @param {String} str the `String` to be used for this `Simple`
   * @returns {baja.DefaultSimple} an instance of the `Simple`
   */
  DefaultSimple.prototype.make = function (str) {
    return this.decodeFromString(str);
  };
  
  /**
   * Decode a `DefaultSimple` from a `String`.
   *
   * @param {String} str
   * @returns {baja.DefaultSimple}
   */
  DefaultSimple.prototype.decodeFromString = function (str) {
    var s = new DefaultSimple(str);
    s.getType = this.getType;
    return s;
  };
  
  /**
   * Encode a `DefaultSimple` to a `String`.
   *
   * @returns {String}
   */
  DefaultSimple.prototype.encodeToString = function () {
    return this.$val;
  };

  /**
   * Returns the String representation of this object.
   *
   * @see baja.Simple#toString
   *
   * @returns {String}
   */
  DefaultSimple.prototype.toString = function () {
    if (this.$displayValue !== undefined) {
      return this.$displayValue;
    } else {
      return this.$val;
    }
  };

  return DefaultSimple;
});
