/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Raymond A. Richards
 */

/**
 * Defines {@link baja.LinkCheck}.
 * @module baja/comp/LinkCheck
 */
define([

    ], function (
    ) {

  "use strict";

  /**
   * LinkCheck is the result of a call to Component.checkLink().
   * It stores information about a link's validity, and if the LinkCheck
   * is invalid then it stores the reason.
   *
   * @param {bool} valid
   * @param {String} reason
   * @constructor
   */
  var LinkCheck = function LinkCheck(valid, reason) {
      this.valid = valid;
      this.invalidReason = reason;
  };

  /**
   * Indicates the validity of the LinkCheck
   * @returns {bool} true if the LinkCheck is valid
   */
  LinkCheck.prototype.isValid = function() {
      return this.valid;
  };

  /**
   * Returns the reason a LinkCheck is invalid.
   * @returns {String} the reason the LinkCheck is invalid.
   */
  LinkCheck.prototype.getInvalidReason = function() {
      return this.invalidReason;
  };

  /**
   * Make a LinkCheck indicating a valid link.
   * @returns {LinkCheck}
   */
  LinkCheck.makeValid = function() {
      return new LinkCheck(true, undefined);
  };

  /**
   * Make a LinkCheck indicating an invalid link.
   * This invalidReason should be localized for the Context passed to
   * {@code Component.linkCheck()}.
   *
   * @param {String} invalidReason
   * @returns {LinkCheck}
   */
  LinkCheck.makeInvalid = function(invalidReason) {
    return new LinkCheck(false, invalidReason);
  };

  /**
   * Decode a LinkCheck from JSON.
   *
   * @see BComponentSpaceSessionHandler.checkLink()
   *
   * @param {String} resp JSON encoded LinkCheck
   * @returns {LinkCheck}
   */
  LinkCheck.fromJSON = function(resp) {
      return new LinkCheck(resp.v, resp.v ? undefined : resp.r);
  };

  return LinkCheck;
});