/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

define(["bajaScript/baja/sys/inherit",
        "bajaScript/baja/sys/bajaUtils",
        "bajaScript/baja/comm/ServerError"], function (
         inherit,
         bajaUtils,
         ServerError) {
  "use strict";
  
  var subclass = inherit.subclass,
      defaultErrorMessage = "Default Message";

  /**
   * A BOX Error.
   *
   * A BOX Error is one that originated from the Server using the BOX protocol.
   *
   * @class
   * @alias baja.comm.BoxError
   * @private
   * @extends baja.comm.ServerError
   * @param [errorType] the type of error
   * @param [obj]
   * @param [obj.c] the error class name
   * @param [obj.m] the error message
   * @param [obj.s] the error stack trace
   */
  var BoxError = function (errorType, obj) {
    obj = obj || {};

    ServerError.call(this, obj.m || defaultErrorMessage);
    this.name = errorType;


    /**
     * The name of the Java class that originated the error station-side
     * @type {String}
     */
    this.javaClass = obj.c;

    /**
     * The full stack trace of the Java exception station-side
     * @type {String}
     */
    this.javaStackTrace = obj.s;
  };
   
  subclass(BoxError, ServerError);



  /**
   * decodeFromServer properly decodes the escaped error message and stack trace
   * to create a new BoxError.
   * @param [errorType] the type of error
   * @param [obj]
   * @param [obj.c] the error class name
   * @param [obj.m] the escaped error message
   * @param [obj.s] the escaped error stack trace
   * @return {baja.comm.BoxError}
   */
  BoxError.decodeFromServer = function (errorType, obj) {

    if (obj && obj.s) {
      obj.s = bajaUtils.unescapeXml(obj.s);
    }

    if (obj && obj.m) {
      obj.m = bajaUtils.unescapeXml(obj.m);
    }
    return new BoxError(errorType, obj);
  };

  return BoxError;
});
