/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * Defines `TableCursor` (not exposed on `baja` namespace).
 * @module baja/coll/TableCursor
 */
define(["bajaScript/sys",
        "bajaScript/baja/coll/QueryCursor",
        "bajaScript/baja/coll/collUtil"], 
        function (baja, QueryCursor, collUtil) {
  
  "use strict";
  
  var subclass = baja.subclass,
      callSuper = baja.callSuper,
      
      colGet = collUtil.colGet;
  
  /**
   * Cursor for a `Table`.
   * 
   * @see baja.coll.Table
   *
   * @class
   * @alias module:baja/coll/TableCursor
   * @extends module:baja/coll/QueryCursor
   * @inner
   * @public
   */
  var TableCursor = function TableCursor(source, curData) {
    callSuper(TableCursor, this, arguments);
    this.$source = source;
  };
  
  subclass(TableCursor, QueryCursor);
  
  function getColDisp(cursor, column, display) { 
    var tof = typeof column,
        row;
    
    if (tof === "undefined") {
      // If no column is defined then just return the entire row
      return colGet(cursor);
    }
    
    if (column && tof === "object") {
      column = column.getName();
    }
    else if (tof !== "string") {
      throw new Error("Invalid Column name: " + column);
    }
    
    row = colGet(cursor);
    if (row !== null) {
      return display ? row.getDisplay(column) : row.get(column);
    }
    
    return null;
  }
        
  /**
   * Return the current row or row item. 
   * 
   * If column information is passed into this method then the value for a particular
   * column and row will be returned.
   *
   * @param {String|module:baja/coll/Table.TableColumn} [column] the column 
   * name or column. If undefined, the entire row is returned.
   * @returns the cursor value (null if none available).
   */
  TableCursor.prototype.get = function (column) { 
    return getColDisp(this, column, /*display*/false);
  }; 
  
  /**
   * Return the current item display string.
   * 
   * If column information is passed into this method then the display String for a particular
   * column and row will be returned.
   * 
   * @param {String|module:baja/coll/Table.TableColumn} [column] the column 
   * name or column. If undefined, the entire row is returned.
   * @returns {String} the cursor display string (null if none available).
   */
  TableCursor.prototype.getDisplay = function (column) {
    return getColDisp(this, column, /*display*/true);
  }; 
  
  return TableCursor;
});