/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * @module baja/boxcs/FireTopicOp
 * @private
 */
define(["bajaScript/bson",
        "bajaScript/baja/boxcs/SyncOp"], 
        function (baja, SyncOp) {
  
  "use strict";
  
  var subclass = baja.subclass,
      callSuper = baja.callSuper,
      
      serverDecodeContext = baja.$serverDecodeContext,
      importUnknownTypes = baja.bson.importUnknownTypes,
      bsonDecodeValue = baja.bson.decodeValue,
      bsonEncodeValue = baja.bson.encodeValue;
  
  /**
   * @class Fire Topic SyncOp.
   *
   * @name FireTopicOp
   * @extends SyncOp
   * @inner
   * @private
   *
   * @param comp the Component the Topic is being fired from.
   * @param {baja.Slot} slot the Topic Slot.
   * @param event the event to be fired (can be null).
   */ 
  function FireTopicOp(comp, slot, event) {
    callSuper(FireTopicOp, this, arguments);
    
    // Encode argument to a data structure
    var a = {
      nm: this.getId(),
      h: comp.getHandle(),
      n: slot.getName()
    };
    
    if (event !== null) {
      a.b = bsonEncodeValue(event); // Encode the value to BSON
    }
    
    this.$arg = a;     
  }
  
  subclass(FireTopicOp, SyncOp); 
  
  FireTopicOp.id = "t";
  
  function fireDecodedTopic(slot, comp, event, display) {
    // Fire the Topic on the Component
    comp.fire({
      "slot": slot, 
      "value": event, 
      "cx": { commit: true, display: display, serverDecode: true }
    }); 
  }
  
  /**
   * Decode and commit the SyncOp.
   *
   * @private
   *
   * @param comp the Component the Topic is being fired from.
   * @param sp the syncOp data structure to be decoded.
   */
  FireTopicOp.decodeAndCommit = function (comp, sp) {  
    if (comp !== null) {  
      // Decode and fire the Component event        
      // TODO: Propagate amongst Knobs
      
      var name = sp.n,
          slot = comp.getSlot(name),
          display = "",
          event = null;
       
      if (slot) {       
        if (sp.b) {
          return importUnknownTypes(sp.b)
            .then(function () {
              event = bsonDecodeValue(sp.b, serverDecodeContext);
              display = sp.b.d;

              fireDecodedTopic(slot, comp, event, display);
            });
        }
        else {
          fireDecodedTopic(slot, comp, event, display);
        }
      }
    }
  };
  
  return FireTopicOp;
});