/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * @module baja/boxcs/BoxCallbacks
 * @private
 */
define([
  "bajaScript/bson",
  "bajaScript/baja/boxcs/AddOp",
  "bajaScript/baja/boxcs/FireTopicOp",
  "bajaScript/baja/boxcs/RemoveOp",
  "bajaScript/baja/boxcs/RenameOp",
  "bajaScript/baja/boxcs/ReorderOp",
  "bajaScript/baja/boxcs/SetOp",
  "bajaScript/baja/boxcs/SetFacetsOp",
  "bajaScript/baja/boxcs/SetFlagsOp",
  "bajaScript/baja/tag/entityDecoder",
  "bajaScript/baja/comp/LinkCheck",
  "bajaScript/baja/comm/Callback" ], function (
    baja,
    AddOp,
    FireTopicOp,
    RemoveOp,
    RenameOp,
    ReorderOp,
    SetOp,
    SetFacetsOp,
    SetFlagsOp,
    entityDecoder,
    LinkCheck,
    Callback) {
  
  "use strict";
  
  var strictArg = baja.strictArg,
      serverDecodeContext = baja.$serverDecodeContext,
      importUnknownTypes = baja.bson.importUnknownTypes,
      bsonDecodeValue = baja.bson.decodeValue;
  
  /**
   * BOX Callbacks plugs into a Component Space so it can make network calls.
   *
   * @private
   * @class
   * @name BoxCallbacks
   * @param {baja.ComponentSpace} space
   */
  function BoxCallbacks(space) {
    this.$space = space;
  }
  
  /**
   * Load Slots.
   *
   * @private
   *
   * @param {String} ord
   * @param {Number} depth
   * @param {baja.comm.Callback} cb
   */
  BoxCallbacks.prototype.loadSlots = function (ord, depth, cb) {
    // Intermediate callback
    var arg = { // Load Slots Argument
          o: ord,
          d: depth
        },
        failedCalled = false,
        loadSlotsCb;
    
    // Only call the ok if fail hasn't been called on the syncTo handler
    cb.addOk(function (ok /*, fail, resp */) {
      if (!failedCalled) {
        // Pass results of the committed syncOps into the result of the original callback
        ok();      
      }
    });
        
    loadSlotsCb = new Callback(baja.ok, cb.fail, cb.getBatch());

    // Mark that fail has been called
    loadSlotsCb.addFail(function (ok, fail, err) {
      failedCalled = true;
      fail(err);
    });

    serverCall(this, "loadSlots", arg, loadSlotsCb);

    // Make a component space sync    
    this.poll(cb);
  };
  
  /**
   * Load Slot Path.
   *
   * @private
   *
   * @param {Array} slotPathInfo
   * @param container
   * @param {baja.comm.Callback} cb
   */
  BoxCallbacks.prototype.loadSlotPath = function (slotPathInfo, container, cb) {
    var space = this.$space,
        arg;
    
    // Intermediate callback
    cb.addOk(function (ok, fail, resp) {
    
      function newOk() {  
        // Attempt to load the Slot in the Space
        space.$fw("commitSlotInfo", resp); 
        ok();
      }
      
      function newFail(err) {
        fail(err);
      }
      
      if (resp) { 
        importUnknownTypes(resp, newOk, newFail);      
      } else {
        newOk();
      }
    });
    
    // Build up the argument to send to the Server
    arg = {
      spi: slotPathInfo,
      bo: String(container.getNavOrd())
    };

    serverCall(this, "loadSlotPath", arg, cb);
  };
    
  /**
   * Component Subscription.
   *
   * @private
   *
   * @param {Array} ords an array of ORDs to Components.
   * @param {baja.comm.Callback} cb
   */
  BoxCallbacks.prototype.subscribe = function (ords, cb) {
    strictArg(ords, Array);
        
    // Check there is something to subscribe too
    if (ords.length === 0) {
      throw new Error("Cannot Subscribe: nothing to subscribe too");
    }
                  
    // Intermediate callback
    var space = this.$space;
    cb.addOk(function (ok, fail, resp) {
      // Commit the sync ops
      space.$fw("commitSyncOps", resp.e, function () {
        ok(resp.h, resp.ts);
      });
    });

    serverCall(this, "sub", ords, cb);
  };
  
  /**
   * Component Unsubscription.
   *
   * @private
   *
   * @param {Array} ords an array of Components to unsubscribe.
   * @param {baja.comm.Callback} cb
   */
  BoxCallbacks.prototype.unsubscribe = function (ords, cb) {
    strictArg(ords, Array);
        
    // Check there is something to subscribe too
    if (ords.length === 0) {
      throw new Error("Cannot Unsubscribe: nothing to unsubscribe too");
    }
       
    function newOk(comp) {
      try {
        comp.$fw("fwUnsubscribed");
      } catch (err) {
        baja.error(err);
      }
    }
       
    // Intermediate callback
    var space = this.$space;
    
    cb.addOk(function (ok/*, fail, resp*/) {
      // Call unsubscribed callbacks
      var i;
      for (i = 0; i < ords.length; ++i) {
        baja.Ord.make(ords[i]).get({
          ok: newOk,
          base: space
        });
      }
      
      ok();      
    });

    serverCall(this, "unsub", ords, cb);
  };
  
  /**
   * Invoke an Action.
   *
   * @private
   *
   * @param comp the Component the Action will be invoked upon.
   * @param {baja.Slot} action the Action Slot.
   * @param val the argument for the Action (can be null).
   * @param {baja.comm.Callback} cb
   */
  BoxCallbacks.prototype.invokeAction = function (comp, action, val, cb) {
        
    // Intermediate callback
    cb.addOk(function (ok, fail, resp) {
      // Decode the value returned
      if (resp !== null) {
        importUnknownTypes(resp, function () {
          // TODO: scan response for unknown Types once batch end callback is fixed     
          ok(bsonDecodeValue(resp, serverDecodeContext));
        }, fail);
      } else {
        ok(null);
      }
    });
    
    var arg = {
      h: comp.getHandle(),
      a: action.getName()
    };
    
    // Encode value if available
    if (val !== null) {
      arg.b = baja.bson.encodeValue(val);
    }

    serverCall(this, "invokeAction", arg, cb);
  };
  
  /**
   * Get the Action Parameter Default Value.
   *
   * @private
   *
   * @param {baja.Component} comp
   * @param {baja.Action} action
   * @param {baja.comm.Callback} cb
   */
  BoxCallbacks.prototype.getActionParameterDefault = function (comp, action, cb) {    
    // Intermediate callback to decode Action parameter default
    cb.addOk(function (ok, fail, resp) {
    
      if (resp !== null) {
        importUnknownTypes(resp, function () {
          ok(bsonDecodeValue(resp, serverDecodeContext));    
        }, fail);
      } else {
        ok(null);
      }
    });

    serverCall(this, "getActionParameterDefault",
      { "h": comp.getHandle(), "a": action.getName() }, cb);
  };
  
  /**
   * Invoke a Server Side Call.
   *
   * @private
   *
   * @param comp the Component for the Server Side Call.
   * @param {String} typeSpec
   * @param {String} methodName
   * @param val the argument for the Server Side Call (can be null).
   * @param {baja.comm.Callback} cb
   */
  BoxCallbacks.prototype.serverSideCall = function (comp, typeSpec, methodName, val, cb) {        
    // Add intermediate callback
    cb.addOk(function (ok, fail, resp) {
      if (resp !== null) {
        importUnknownTypes(resp, function () {
          ok(bsonDecodeValue(resp, serverDecodeContext));
        }, fail);
      } else {
        ok(null);
      }
    });
    
    // Arguments    
    var arg = {
      h: comp.getHandle(),
      ts: typeSpec,
      m: methodName
    };
    
    // Encode value if available
    if (val !== null) {
      arg.b = baja.bson.encodeValue(val);
    }

    serverCall(this, "serverSideCall", arg, cb);
  };

  /**
   * Resolve the enabled mix-ins for the Component Space.
   *
   * @private
   *
   * @param {baja.comm.Callback} cb
   */
  BoxCallbacks.prototype.toEnabledMixIns = function (cb) {
    serverCall(this, "enabledMixIns", {}, cb);
  };
    
  /**
   * Poll the Server for events.
   *
   * @private
   *
   * @param {baja.comm.Callback} cb
   */  
  BoxCallbacks.prototype.poll = function (cb) {    
    baja.comm.poll(cb);
  };
  
  /**
   * Convert a handle to a Slot Path.
   *
   * @private
   *
   * @param {String} handle
   * @param {baja.comm.Callback} cb
   */
  BoxCallbacks.prototype.handleToPath = function (handle, cb) {   
    // Intermediate callback to pass in SlotPath to callback
    cb.addOk(function (ok, fail, slotPathStr) {
      ok(new baja.SlotPath(slotPathStr));
    });

    serverCall(this, "handleToPath", handle, cb);
  };
  
  /**
   * Resolve a Service via its TypeSpec (moduleName:typeName).
   *
   * @private
   *
   * @param {String} typeSpec
   * @param {baja.comm.Callback} cb
   */
  BoxCallbacks.prototype.getService = function (typeSpec, cb) {
    strictArg(typeSpec, String);
    
    // Intermediate callback to resolve the SlotPath into the target Component
    var that = this;
    cb.addOk(function (ok, fail, slotPath) {
      // Resolve the SlotPath ORD
      baja.Ord.make(slotPath.toString()).get({
        "base": that.$space, 
        "ok": ok, 
        "fail": fail
      });
    });
  
    this.serviceToPath(typeSpec, cb);
  };
  
  /**
   * Resolve a Service to its SlotPath via a TypeSpec.
   *
   * @private
   *
   * @param {String} typeSpec
   * @param {baja.comm.Callback} cb
   */
  BoxCallbacks.prototype.serviceToPath = function (typeSpec, cb) {    
    // Intermediate callback to pass in SlotPath to callback
    cb.addOk(function (ok, fail, slotPathStr) {
      ok(new baja.SlotPath(slotPathStr));
    });

    serverCall(this, "serviceToPath", typeSpec, cb);
  };
  
  /**
   * Make a Link.
   *
   * @private
   *
   * @param {baja.Component} source Component for the link.
   * @param {baja.Slot} sourceSlot source Slot for the link.
   * @param {baja.Component} target Component for the link.
   * @param {baja.Slot} targetSlot target Slot for the link.
   * @param {baja.comm.Callback} cb 
   */
  BoxCallbacks.prototype.makeLink = function (source, sourceSlot, target, targetSlot, cb) {     
    // Add intermediate callback
    cb.addOk(function (ok, fail, resp) {    
      importUnknownTypes(resp, function () {
        ok(bsonDecodeValue(resp, serverDecodeContext));
      }, fail);
    });
    
    // Arguments    
    var arg = {
      s: source.getHandle(),
      ss: sourceSlot.getName(),
      t: target.getHandle(),
      ts: targetSlot.getName()
    };

    serverCall(this, "makeLink", arg, cb);
  };

  /**
   * Check a Link.
   *
   * @private
   *
   * @param {baja.Component} source Component for the link.
   * @param {baja.Slot} sourceSlot source Slot for the link.
   * @param {baja.Component} target Component for the link.
   * @param {baja.Slot} targetSlot target Slot for the link.
   * @param {baja.comm.Callback} cb
   */
  BoxCallbacks.prototype.checkLink = function (source, sourceSlot, target, targetSlot, cb) {
    // Add intermediate callback to decode the LinkCheck
    cb.addOk(function (ok, fail, resp) {
      ok(LinkCheck.fromJSON(resp));
    });

    // Arguments
    var arg = {
      s: source.getHandle(),
      ss: sourceSlot.getName(),
      t: target.getHandle(),
      ts: targetSlot.getName()
    };

    serverCall(this, "checkLink", arg, cb);
  };

  /**
   * Get the Nav Children of a Component.
   *
   * @private
   *
   * @param {String} handle
   * @param {baja.comm.Callback} cb
   */
  BoxCallbacks.prototype.getNavChildren = function (handle, cb) {   
    // Intermediate callback to resolve Nav ORDs
    cb.addOk(function (ok, fail, navOrds) {
      // Resolve each of the Nav ORDs
      new baja.BatchResolve(navOrds).resolve({
        ok: function () {
          // Pass the resolved Components to the callback handler
          ok(this.getTargetObjects());
        },
        fail: fail
      });
    });

    serverCall(this, "navChildren", handle, cb);
  };

  /**
   * Get the implied Tags for a Component.
   *
   * @private
   *
   * @param {String} handle
   * @param {baja.comm.Callback} cb
   */
  BoxCallbacks.prototype.getImpliedTags = function (handle, cb) {   
    // Intermediate callback to decode the Tags JSON.
    cb.addOk(function (ok, fail, tagsJson) {
      ok(entityDecoder.decodeTags(tagsJson));
    });

    serverCall(this, "impliedTags", handle, cb);
  };

  /**
   * Get the implied Relations for a Component.
   *
   * @private
   *
   * @param {String} handle
   * @param {baja.comm.Callback} cb
   */
  BoxCallbacks.prototype.getImpliedRelations = function (handle, cb) {   
    // Intermediate callback to decode the relations JSON.
    cb.addOk(function (ok, fail, relationsJson) {
      ok(entityDecoder.decodeRelations(relationsJson));
    });

    serverCall(this, "impliedRelations", handle, cb);
  };
    
  ////////////////////////////////////////////////////////////////
  // sendToMaster SyncOps
  //////////////////////////////////////////////////////////////// 
  
  /**
   * Server Add.
   *
   * @private
   *
   * @param comp the Component being added too.
   * @param {String} slotName
   * @param val the value to be added.
   * @param {Number} flags slot flags.
   * @param {baja.Facets} facets slot facets.
   * @param {baja.comm.Callback} cb   
   */
  BoxCallbacks.prototype.add = function (comp, slotName, val, flags, facets, cb) {
    // Add intermediate callback to pass back newly added Property
    cb.addOk(function (ok, fail, resp) {
      // Attempt to get newly added Property name from server response
      var newName = slotName;
      if (resp && resp instanceof Array && resp.length > 0 && resp[0].nn) {
        newName = resp[0].nn;
      }
    
      // Please note: if the slot name had a wildcard in it, this won't work (i.e. 'test?')
      ok(comp.getSlot(newName));
    });
  
    // Send the op to the Server
    new AddOp(comp, slotName, val, flags, facets).syncTo(comp.getComponentSpace(), cb);   
  };  
  
  /**
   * Server Set.
   *
   * @private
   *
   * @param comp the Component being added too.
   * @param {Array} propPath array of Property names used for the set.
   * @param val the value for the set.
   * @param {baja.comm.Callback} cb   
   */
  BoxCallbacks.prototype.set = function (comp, propPath, val, cb) {     
    // Send the op to the Server
    new SetOp(comp, propPath, val).syncTo(comp.getComponentSpace(), cb);   
  };  
  
  /**
   * Server Remove.
   *
   * @private
   *
   * @param comp the Component being removed from.
   * @param {baja.Slot} slot the slot to be removed.
   * @param {baja.comm.Callback} cb   
   */
  BoxCallbacks.prototype.remove = function (comp, slot, cb) {     
    // Send the op to the Server
    new RemoveOp(comp, slot).syncTo(comp.getComponentSpace(), cb);   
  };  
  
  /**
   * Server Rename.
   *
   * @private
   *
   * @param comp the Component the slot is being renamed on.
   * @param {String} oldName the old name of the slot.
   * @param {String} newName the new name of the slot.
   * @param {baja.comm.Callback} cb
   */
  BoxCallbacks.prototype.rename = function (comp, oldName, newName, cb) {     
    // Send the op to the Server
    new RenameOp(comp, oldName, newName).syncTo(comp.getComponentSpace(), cb);   
  };
  
  /**
   * Server Reorder.
   *
   * @private
   *
   * @param comp the Component the dynamic slots are being reordered upon.
   * @param {Array} dynamicProperties an array of Property names that specify the new order.
   * @param {baja.comm.Callback} cb
   */
  BoxCallbacks.prototype.reorder = function (comp, dynamicProperties, cb) {     
    // Send the op to the Server
    new ReorderOp(comp, dynamicProperties).syncTo(comp.getComponentSpace(), cb);   
  };
  
  /**
   * Server Set Flags.
   *
   * @private
   *
   * @param comp the Component for the slot the flags will be set upon.
   * @param {baja.Slot} slot the slot the flags are being set upon.
   * @param {Number} flags the new slot flags.
   * @param {baja.comm.Callback} cb 
   */
  BoxCallbacks.prototype.setFlags = function (comp, slot, flags, cb) {     
    // Send the op to the Server
    new SetFlagsOp(comp, slot, flags).syncTo(comp.getComponentSpace(), cb);   
  };
  
  /**
   * Server Set Facets.
   *
   * @private
   *
   * @param comp the Component for the slot the facets will be set upon.
   * @param {baja.Slot} slot the dynamic slot the facets are being set upon.
   * @param {baja.Facets} facets the new slot facets.
   * @param {baja.comm.Callback} cb 
   */
  BoxCallbacks.prototype.setFacets = function (comp, slot, facets, cb) {     
    // Send the op to the Server
    new SetFacetsOp(comp, slot, facets).syncTo(comp.getComponentSpace(), cb);   
  };
    
  /**
   * Server Topic Fire.
   *
   * @private
   *
   * @param comp the Component the Topic will be fired from.
   * @param {baja.Slot} slot the Topic Slot.
   * @param event the Topic event (can be null).
   * @param {baja.comm.Callback} cb 
   */
  BoxCallbacks.prototype.fire = function (comp, slot, event, cb) {     
    // Send the op to the Server
    new FireTopicOp(comp, slot, event).syncTo(comp.getComponentSpace(), cb);   
  };


  /**
   * @param {BoxCallbacks} boxCallbacks
   * @param {string} key
   * @param {*} arg
   * @param {baja.comm.Callback} cb
   */
  function serverCall(boxCallbacks, key, arg, cb) {
    baja.comm.serverHandlerCall(boxCallbacks.$space, key, arg, cb);
  }
  
  return BoxCallbacks;
});
