/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 */

/* jshint browser: true */

/**
 * API Status: **Private**
 * @module nmodule/bacnet/rc/wb/mgr/model/columns/MacAddressMgrColumn
 */
define(['baja!',
        'underscore',
        'Promise',
        'nmodule/bacnet/rc/baja/datatypes/BacnetAddress',
        'nmodule/bacnet/rc/baja/datatypes/BacnetOctetString',
        'nmodule/webEditors/rc/wb/mgr/model/columns/PropertyPathMgrColumn',
        'nmodule/webEditors/rc/wb/mgr/model/MgrColumn',
        'nmodule/js/rc/log/Log',
        'baja!bacnet:BacnetDeviceObject'], function (
        baja,
        _,
        Promise,
        BacnetAddress,
        BacnetOctetString,
        PropertyPathMgrColumn,
        MgrColumn,
        Log) {

  'use strict';

  var MAC_TYPE_UNKNOWN = 0,
      MAC_TYPE_ETHERNET = 1,
      MAC_TYPE_IP = 2,
      MAC_TYPE_MSTP = 3;

  /**
   * Column type for the device's MAC address in the main database table.
   * This works as a string editor, converting it to an octet string at
   * validation and commit time. This allows it to load the value for a
   * mis-configured device, without an octet string editor immediately
   * rejecting an invalid address.
   */
  var MacAddressMgrColumn = function MacAddressMgrColumn (name, params) {
    PropertyPathMgrColumn.call(this, name, params);
  };

  MacAddressMgrColumn.prototype = Object.create(PropertyPathMgrColumn.prototype);
  MacAddressMgrColumn.prototype.constructor = MacAddressMgrColumn;

  /**
   * Returns the address as a formatted string.
   *
   * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row
   * @returns {String}
   */
  MacAddressMgrColumn.prototype.getValueFor = function (row) {
    var address = row.getSubject().getAddress(),
        mac = address.getMacAddress();

    try {
      return BacnetAddress.bytesToString(address.getAddressType(), mac.getBytes());
    }
    catch (e) {
      Log.logMessage('MacAddressMgrColumn', Log.Level.FINE, 'Could not parse address: ' + String(e));

      return '';
    }
  };

  /**
   * Examine the given string and try to determine what type of MAC type
   * it represents.
   *
   * @param {String} str - the value from the editor
   * @returns {number} - one of the MAC_TYPE_* values
   */
  MacAddressMgrColumn.$getAddressType = function (str) {
    if (str.indexOf('.') > 0) { return MAC_TYPE_IP; }
    else if (str.indexOf(':') > 0) { return MAC_TYPE_ETHERNET; }
    else if ((str !== 'null') && (str.length <= 4)) { return MAC_TYPE_MSTP; }
    else { return MAC_TYPE_UNKNOWN; }
  };

  /**
   * Parse the given string according to the MAC type and return the
   * array of byte values it represents.
   *
   * @param {Number} type - a MAC type
   * @param {String} str - the value from the editor
   * @returns {Array} - a byte array parsed from the string
   */
  MacAddressMgrColumn.$macStringToBytes = function (type, str) {
    var len = 0;

    if (type === MAC_TYPE_IP) { len = (str.indexOf(':') > -1) ? 5 : 4; }
    return BacnetAddress.stringToBytes(type, len, str);
  };

  /**
   * Test whether the proposed value can be parsed to a valid byte
   * array and, if not, then reject.
   *
   * @param {module:nmodule/webEditors/rc/wb/mgr/model/MgrModel} model
   * @param {Array} data - the array of proposed values
   * @param {Object} [params]
   * @returns {Promise.<*>}
   */
  MacAddressMgrColumn.prototype.mgrValidate = function (model, data, params) {
    var addressType, value;

    for (var i = 0; i < data.length; i++) {
      try {
        value = data[i];

        if (value !== null) {
          addressType = MacAddressMgrColumn.$getAddressType(value);
          MacAddressMgrColumn.$macStringToBytes(addressType, value);
        }
      }
      catch (e) {
        return Promise.reject(new Error(e));
      }
    }

    return Promise.resolve();
  };

  /**
   * Save the value back to the 'addressType' and 'macAddress' properties
   * of the device's 'address' property.
   *
   * @param {String} value
   * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row
   * @param {object} [params]
   * @returns {Promise.<*>}
   */
  MacAddressMgrColumn.prototype.commit = function (value, row, params) {
    var device = row.getSubject(),
        address = device.getAddress(),
        batch = params && params.batch,
        progressCallback = params && params.progressCallback,
        addressType,
        mac,
        prom;

    addressType = MacAddressMgrColumn.$getAddressType(value);
    mac = MacAddressMgrColumn.$macStringToBytes(addressType, value);

    // Note that we want to set the two properties on the address object,
    // but don't replace the entire address object, as we don't want to overwrite
    // the network number.

    prom = Promise.all([
      address.set({
        slot: 'addressType', value: addressType, batch: batch
      }),
      address.set({
        slot: 'macAddress', value: BacnetOctetString.make(mac), batch: batch
      })
    ]);

    if (progressCallback) { progressCallback(MgrColumn.COMMIT_READY); }

    return prom;
  };

  return MacAddressMgrColumn;
});
