/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 */

/* jshint browser: true */

/**
 * API Status: **Private**
 * @module nmodule/bacnet/rc/wb/mgr/commands/WhoHasCommand
 */
define(['baja!',
        'lex!bacnet',
        'jquery',
        'Promise',
        'underscore',
        'dialogs',
        'bajaux/commands/Command',
        'nmodule/webEditors/rc/fe/feDialogs',
        'nmodule/webEditors/rc/wb/PropertySheet',
        'nmodule/webEditors/rc/wb/mgr/commands/MgrCommand',
        'baja!bacnet:WhoHasConfig'], function (
        baja,
        lexs,
        $,
        Promise,
        _,
        dialogs,
        Command,
        feDialogs,
        PropertySheet,
        addMgrCommandSupport) {

  'use strict';

  var bacnetLex = lexs[0];

  /**
   * Command to search the network for devices containing objects with a particular
   * id or name. This will query the network and display the results in a dialog
   * when the job complete event is fired.
   *
   * @class
   * @alias module:nmodule/bacnet/rc/wb/mgr/commands/WhoHasCommand
   * @extends module:bajaux/commands/Command
   */
  var WhoHasCommand = function WhoHasCommand(mgr) {
    var that = this;

    Command.call(that, {
      displayName: bacnetLex.get('whoHas.label'),
      icon: bacnetLex.get('whoHas.icon'),
      enabled: true,

      /**
       * Show the configuration property sheet, then pass the configuration
       * parameters to the action to start the job.
       *
       * @alias module:nmodule/bacnet/rc/wb/mgr/commands/WhoHasCommand#invoke
       * @returns {Promise}
       */
      func: function () {
        var network = mgr.getNetwork();

        return mgr.makeDiscoveryNetworks(network)
          .then(function (discoveryNetworks) {
            var config = baja.$('bacnet:WhoHasConfig');
            config.setNetworks(discoveryNetworks);

            return feDialogs.showFor({
              title: bacnetLex.get('whoHas.description'),
              type: PropertySheet,
              value: config,
              properties: {
                showHeader: false,
                showFooter: false
              }
            })
              .then(function (diff) {
                if (diff) {
                  return diff.apply(config)
                    .then(function () {
                      return network.submitDeviceManagerJob(config);
                    })
                    .then(function (ord) {
                      mgr.setJob(baja.Ord.make({
                        base: baja.Ord.make('station:'),
                        child: ord.relativizeToSession()
                      }));
                    });
                }
              });
          });
      }
    });

    addMgrCommandSupport(that);
    that.setShowInActionBar(false);

    mgr.on('jobcomplete', function (job) {
      if (job.getType().is('bacnet:WhoHasJob')) {
        that.complete(job);
      }
    });
  };

  WhoHasCommand.prototype = Object.create(Command.prototype);
  WhoHasCommand.prototype.constructor = WhoHasCommand;

  /**
   * Function called when the 'who has' job has completed. This will get the encoded
   * 'I Have' responses from the job and display the results in a dialog.
   *
   * @param {baja.Component} job - the 'who has' job, submitted by the invoke function
   * above.
   */
  WhoHasCommand.prototype.complete = function (job) {
    return job.loadSlots()
      .then(function () {
        var slots = job.getSlots().properties().is('baja:String').toValueArray(),
            txt = 'I-Have Responses for <br>' + slots[0] + '<br><ul>';

        _.each(_.tail(slots), function (str) {
          var tokens = str.split('|');
          txt = txt + '<li>' + tokens[0] + ' has object ' + tokens[1] + ' named ' + tokens[2] + '</li>';
        });

        txt = txt + '</ul>';

        dialogs.showOk({
          title: bacnetLex.get('whoHas.label'),
          content: txt
        });
      });
  };

  return WhoHasCommand;
});
